<?php
require_once 'config/config.php';

if (isLoggedIn()) {
    redirect(SITE_URL . '/account.php');
}

$error = '';
$success = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $name = sanitize($_POST['name'] ?? '');
    $email = sanitize($_POST['email'] ?? '');
    $password = $_POST['password'] ?? '';
    $confirmPassword = $_POST['confirm_password'] ?? '';

    if ($name && $email && $password && $confirmPassword) {
        if ($password !== $confirmPassword) {
            $error = 'Passwords do not match';
        } elseif (strlen($password) < 6) {
            $error = 'Password must be at least 6 characters long';
        } else {
            $conn = getDBConnection();
            
            // Check if email already exists
            $checkStmt = $conn->prepare("SELECT id FROM users WHERE email = ?");
            $checkStmt->bind_param("s", $email);
            $checkStmt->execute();
            $checkResult = $checkStmt->get_result();
            
            if ($checkResult->num_rows > 0) {
                $error = 'Email already registered';
            } else {
                $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
                $stmt = $conn->prepare("INSERT INTO users (name, email, password) VALUES (?, ?, ?)");
                $stmt->bind_param("sss", $name, $email, $hashedPassword);
                
                if ($stmt->execute()) {
                    $userId = $stmt->insert_id;
                    
                    // Auto login
                    $_SESSION['user_id'] = $userId;
                    $_SESSION['user_name'] = $name;
                    $_SESSION['user_email'] = $email;
                    $_SESSION['user_role'] = 'customer';

                    // Merge cart
                    $sessionId = $_SESSION['cart_session_id'] ?? '';
                    if ($sessionId) {
                        $mergeStmt = $conn->prepare("UPDATE cart SET user_id = ?, session_id = NULL WHERE session_id = ?");
                        $mergeStmt->bind_param("is", $userId, $sessionId);
                        $mergeStmt->execute();
                        $mergeStmt->close();
                    }

                    $stmt->close();
                    closeDBConnection($conn);
                    redirect(SITE_URL . '/account.php');
                } else {
                    $error = 'Registration failed. Please try again.';
                }
                $stmt->close();
            }
            $checkStmt->close();
            closeDBConnection($conn);
        }
    } else {
        $error = 'Please fill in all fields';
    }
}

$pageTitle = "Register | " . SITE_NAME;
require_once 'includes/header.php';
?>

<section class="py-20 min-h-screen bg-gray-50">
    <div class="container mx-auto px-4">
        <div class="max-w-md mx-auto">
            <div class="bg-white rounded-2xl shadow-xl p-8">
                <div class="text-center mb-8">
                    <h1 class="text-3xl font-display font-bold text-gray-900 mb-2">Create Account</h1>
                    <p class="text-gray-600">Join us for natural beauty</p>
                </div>

                <?php if ($error): ?>
                    <div class="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-lg mb-6">
                        <i class="fas fa-exclamation-circle mr-2"></i><?php echo $error; ?>
                    </div>
                <?php endif; ?>

                <form method="POST" action="">
                    <div class="mb-4">
                        <label class="block text-gray-700 font-semibold mb-2">Full Name</label>
                        <input type="text" name="name" required 
                               class="input-field" placeholder="John Doe"
                               value="<?php echo htmlspecialchars($_POST['name'] ?? ''); ?>">
                    </div>

                    <div class="mb-4">
                        <label class="block text-gray-700 font-semibold mb-2">Email Address</label>
                        <input type="email" name="email" required 
                               class="input-field" placeholder="your@email.com"
                               value="<?php echo htmlspecialchars($_POST['email'] ?? ''); ?>">
                    </div>

                    <div class="mb-4">
                        <label class="block text-gray-700 font-semibold mb-2">Password</label>
                        <input type="password" name="password" required 
                               class="input-field" placeholder="At least 6 characters">
                    </div>

                    <div class="mb-6">
                        <label class="block text-gray-700 font-semibold mb-2">Confirm Password</label>
                        <input type="password" name="confirm_password" required 
                               class="input-field" placeholder="Re-enter your password">
                    </div>

                    <div class="mb-6">
                        <label class="flex items-start">
                            <input type="checkbox" required class="mr-2 mt-1 rounded">
                            <span class="text-sm text-gray-600">
                                I agree to the <a href="#" class="text-primary-600 hover:text-primary-700">Terms & Conditions</a> 
                                and <a href="#" class="text-primary-600 hover:text-primary-700">Privacy Policy</a>
                            </span>
                        </label>
                    </div>

                    <button type="submit" class="btn-primary w-full text-lg">
                        <i class="fas fa-user-plus mr-2"></i>Create Account
                    </button>
                </form>

                <div class="mt-6 text-center">
                    <p class="text-gray-600">
                        Already have an account?
                        <a href="<?php echo SITE_URL; ?>/login.php" class="text-primary-600 hover:text-primary-700 font-semibold">
                            Login here
                        </a>
                    </p>
                </div>
            </div>
        </div>
    </div>
</section>

<?php require_once 'includes/footer.php'; ?>
