<?php
require_once __DIR__ . '/../config/config.php';

if (!isLoggedIn()) {
    $_SESSION['admin_redirect'] = true;
    redirect(SITE_URL . '/login.php?redirect=admin');
} elseif (!isAdmin()) {
    redirect(SITE_URL . '/index.php');
}

$activePage = 'products';
$pageTitle = 'Manage Products | ' . SITE_NAME;

$conn = getDBConnection();
$errors = [];
$success = '';

// Handle create/update/delete actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';

    if (in_array($action, ['create', 'update'], true)) {
        $productId = intval($_POST['product_id'] ?? 0);
        $name = sanitize($_POST['name'] ?? '');
        $sku = sanitize($_POST['sku'] ?? '');
        $price = floatval($_POST['price'] ?? 0);
        $salePrice = strlen($_POST['sale_price'] ?? '') ? floatval($_POST['sale_price']) : null;
        $categoryId = intval($_POST['category_id'] ?? 0) ?: null;
        $stockQuantity = intval($_POST['stock_quantity'] ?? 0);
        $status = in_array($_POST['status'] ?? 'active', ['active', 'inactive'], true) ? $_POST['status'] : 'active';
        $featured = isset($_POST['featured']) ? 1 : 0;
        $newArrival = isset($_POST['new_arrival']) ? 1 : 0;
        $shortDescription = trim($_POST['short_description'] ?? '');
        $description = trim($_POST['description'] ?? '');

        if (!$name) {
            $errors[] = 'Product name is required.';
        }
        if ($price <= 0) {
            $errors[] = 'Regular price must be greater than zero.';
        }
        if ($salePrice !== null && $salePrice >= $price) {
            $errors[] = 'Sale price must be lower than regular price.';
        }

        $imageFileName = saveUploadedImage($_FILES['image'] ?? [], 'products');

        if (!$errors) {
            if ($action === 'create') {
                $slug = slugify($name);
                $stmt = $conn->prepare("INSERT INTO products (name, slug, description, short_description, price, sale_price, category_id, stock_quantity, sku, featured, new_arrival, status) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
                $stmt->bind_param('sssssdiissis', $name, $slug, $description, $shortDescription, $price, $salePrice, $categoryId, $stockQuantity, $sku, $featured, $newArrival, $status);
                $stmt->execute();
                $newProductId = $stmt->insert_id;
                $stmt->close();

                if ($imageFileName) {
                    $imgStmt = $conn->prepare("INSERT INTO product_images (product_id, image_url, is_primary, display_order) VALUES (?, ?, 1, 0)");
                    $imgStmt->bind_param('is', $newProductId, $imageFileName);
                    $imgStmt->execute();
                    $imgStmt->close();
                }

                $success = 'Product created successfully.';
            } elseif ($action === 'update' && $productId > 0) {
                $slug = slugify($name);
                $stmt = $conn->prepare("UPDATE products SET name = ?, slug = ?, description = ?, short_description = ?, price = ?, sale_price = ?, category_id = ?, stock_quantity = ?, sku = ?, featured = ?, new_arrival = ?, status = ? WHERE id = ?");
                $stmt->bind_param('sssssdiissisi', $name, $slug, $description, $shortDescription, $price, $salePrice, $categoryId, $stockQuantity, $sku, $featured, $newArrival, $status, $productId);
                $stmt->execute();
                $stmt->close();

                if ($imageFileName) {
                    $imgStmt = $conn->prepare("INSERT INTO product_images (product_id, image_url, is_primary, display_order) VALUES (?, ?, 1, 0)");
                    $imgStmt->bind_param('is', $productId, $imageFileName);
                    $imgStmt->execute();
                    $imgStmt->close();
                }

                $success = 'Product updated successfully.';
            }
        }
    } elseif ($action === 'delete') {
        $productId = intval($_POST['product_id'] ?? 0);
        if ($productId) {
            $stmt = $conn->prepare("DELETE FROM products WHERE id = ?");
            $stmt->bind_param('i', $productId);
            $stmt->execute();
            $stmt->close();
            $success = 'Product deleted successfully.';
        }
    }
}

// Fetch supporting data
$categoriesStmt = $conn->query("SELECT id, name FROM categories ORDER BY name ASC");
$categories = $categoriesStmt->fetch_all(MYSQLI_ASSOC);
$categoriesStmt->close();

$productsQuery = "
    SELECT p.*, c.name AS category_name, pi.image_url AS primary_image
    FROM products p
    LEFT JOIN categories c ON p.category_id = c.id
    LEFT JOIN product_images pi ON p.id = pi.product_id AND pi.is_primary = 1
    ORDER BY p.created_at DESC
";
$productsResult = $conn->query($productsQuery);

$editProduct = null;
if (isset($_GET['edit'])) {
    $editId = intval($_GET['edit']);
    $stmt = $conn->prepare("SELECT * FROM products WHERE id = ?");
    $stmt->bind_param('i', $editId);
    $stmt->execute();
    $editProduct = $stmt->get_result()->fetch_assoc();
    $stmt->close();
}

require_once __DIR__ . '/../includes/header.php';
require_once __DIR__ . '/admin_nav.php';
?>

<div class="container mx-auto px-4 pb-12">
    <div class="max-w-5xl mx-auto">
        <div class="bg-white rounded-xl shadow-md border border-gray-200 p-6 mb-8">
            <div class="flex items-center justify-between mb-4">
                <h1 class="text-2xl font-display font-bold text-gray-900"><?php echo $editProduct ? 'Edit Product' : 'Add New Product'; ?></h1>
                <?php if ($editProduct): ?>
                    <a href="<?php echo SITE_URL; ?>/admin/products.php" class="text-sm text-primary-600 hover:text-primary-700">Cancel Edit</a>
                <?php endif; ?>
            </div>

            <?php if ($errors): ?>
                <div class="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-lg mb-4">
                    <ul class="list-disc list-inside text-sm space-y-1">
                        <?php foreach ($errors as $error): ?>
                            <li><?php echo $error; ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            <?php endif; ?>

            <?php if ($success): ?>
                <div class="bg-green-50 border border-green-200 text-green-700 px-4 py-3 rounded-lg mb-4">
                    <i class="fas fa-check-circle mr-2"></i><?php echo $success; ?>
                </div>
            <?php endif; ?>

            <form method="POST" enctype="multipart/form-data" class="space-y-4">
                <input type="hidden" name="action" value="<?php echo $editProduct ? 'update' : 'create'; ?>">
                <?php if ($editProduct): ?>
                    <input type="hidden" name="product_id" value="<?php echo $editProduct['id']; ?>">
                <?php endif; ?>

                <div class="grid md:grid-cols-2 gap-4">
                    <div>
                        <label class="block text-sm font-semibold text-gray-700 mb-2">Product Name</label>
                        <input type="text" name="name" class="input-field" value="<?php echo htmlspecialchars($editProduct['name'] ?? ''); ?>" required>
                    </div>
                    <div>
                        <label class="block text-sm font-semibold text-gray-700 mb-2">SKU</label>
                        <input type="text" name="sku" class="input-field" value="<?php echo htmlspecialchars($editProduct['sku'] ?? ''); ?>" placeholder="Optional">
                    </div>
                </div>

                <div class="grid md:grid-cols-2 gap-4">
                    <div>
                        <label class="block text-sm font-semibold text-gray-700 mb-2">Category</label>
                        <select name="category_id" class="input-field">
                            <option value="">Select category</option>
                            <?php foreach ($categories as $category): ?>
                                <option value="<?php echo $category['id']; ?>" <?php echo (($editProduct['category_id'] ?? '') == $category['id']) ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($category['name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div>
                        <label class="block text-sm font-semibold text-gray-700 mb-2">Stock Quantity</label>
                        <input type="number" name="stock_quantity" min="0" class="input-field" value="<?php echo htmlspecialchars($editProduct['stock_quantity'] ?? 0); ?>">
                    </div>
                </div>

                <div class="grid md:grid-cols-2 gap-4">
                    <div>
                        <label class="block text-sm font-semibold text-gray-700 mb-2">Regular Price (<?php echo CURRENCY; ?>)</label>
                        <input type="number" name="price" min="0" step="0.01" class="input-field" value="<?php echo htmlspecialchars($editProduct['price'] ?? ''); ?>" required>
                    </div>
                    <div>
                        <label class="block text-sm font-semibold text-gray-700 mb-2">Sale Price (Optional)</label>
                        <input type="number" name="sale_price" min="0" step="0.01" class="input-field" value="<?php echo htmlspecialchars($editProduct['sale_price'] ?? ''); ?>">
                    </div>
                </div>

                <div class="grid md:grid-cols-2 gap-4">
                    <div>
                        <label class="block text-sm font-semibold text-gray-700 mb-2">Status</label>
                        <select name="status" class="input-field">
                            <option value="active" <?php echo (($editProduct['status'] ?? '') === 'active') ? 'selected' : ''; ?>>Active</option>
                            <option value="inactive" <?php echo (($editProduct['status'] ?? '') === 'inactive') ? 'selected' : ''; ?>>Inactive</option>
                        </select>
                    </div>
                    <div class="flex items-center gap-6">
                        <label class="inline-flex items-center gap-2">
                            <input type="checkbox" name="featured" <?php echo !empty($editProduct['featured']) ? 'checked' : ''; ?>>
                            <span class="text-sm text-gray-700">Featured</span>
                        </label>
                        <label class="inline-flex items-center gap-2">
                            <input type="checkbox" name="new_arrival" <?php echo !empty($editProduct['new_arrival']) ? 'checked' : ''; ?>>
                            <span class="text-sm text-gray-700">New Arrival</span>
                        </label>
                    </div>
                </div>

                <div>
                    <label class="block text-sm font-semibold text-gray-700 mb-2">Short Description</label>
                    <textarea name="short_description" rows="2" class="input-field" placeholder="Visible on product cards"><?php echo htmlspecialchars($editProduct['short_description'] ?? ''); ?></textarea>
                </div>

                <div>
                    <label class="block text-sm font-semibold text-gray-700 mb-2">Full Description</label>
                    <textarea name="description" rows="4" class="input-field" placeholder="Detailed product information"><?php echo htmlspecialchars($editProduct['description'] ?? ''); ?></textarea>
                </div>

                <div>
                    <label class="block text-sm font-semibold text-gray-700 mb-2">Primary Image</label>
                    <input type="file" name="image" accept="image/jpeg,image/png,image/webp" class="block w-full text-sm text-gray-600">
                    <p class="text-xs text-gray-500 mt-1">Supported formats: JPG, PNG, WEBP. Max 5 MB.</p>
                </div>

                <div class="flex justify-end gap-3">
                    <button type="submit" class="btn-primary">
                        <i class="fas fa-save mr-2"></i>
                        <?php echo $editProduct ? 'Update Product' : 'Create Product'; ?>
                    </button>
                </div>
            </form>
        </div>

        <div class="bg-white rounded-xl shadow-md border border-gray-200 p-6">
            <div class="flex items-center justify-between mb-4">
                <h2 class="text-xl font-display font-bold text-gray-900">Products</h2>
                <span class="text-sm text-gray-500"><?php echo $productsResult->num_rows; ?> total</span>
            </div>

            <div class="overflow-x-auto">
                <table class="w-full text-sm">
                    <thead class="bg-gray-50">
                        <tr>
                            <th class="px-4 py-3 text-left font-semibold text-gray-600">Product</th>
                            <th class="px-4 py-3 text-left font-semibold text-gray-600">Category</th>
                            <th class="px-4 py-3 text-left font-semibold text-gray-600">Price</th>
                            <th class="px-4 py-3 text-left font-semibold text-gray-600">Stock</th>
                            <th class="px-4 py-3 text-left font-semibold text-gray-600">Status</th>
                            <th class="px-4 py-3 text-left font-semibold text-gray-600">Flags</th>
                            <th class="px-4 py-3 text-right font-semibold text-gray-600">Actions</th>
                        </tr>
                    </thead>
                    <tbody class="divide-y divide-gray-100">
                        <?php while ($productRow = $productsResult->fetch_assoc()): 
                            $primaryImage = getProductImage($productRow['primary_image'] ?? null, $productRow['id']);
                        ?>
                            <tr class="hover:bg-gray-50">
                                <td class="px-4 py-3">
                                    <div class="flex items-center gap-3">
                                        <div class="w-12 h-12 rounded-lg overflow-hidden bg-gray-100">
                                            <img src="<?php echo $primaryImage; ?>" alt="<?php echo htmlspecialchars($productRow['name']); ?>" class="w-full h-full object-cover">
                                        </div>
                                        <div>
                                            <p class="font-semibold text-gray-900"><?php echo htmlspecialchars($productRow['name']); ?></p>
                                            <?php if ($productRow['sku']): ?>
                                                <p class="text-xs text-gray-500">SKU: <?php echo htmlspecialchars($productRow['sku']); ?></p>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </td>
                                <td class="px-4 py-3 text-gray-600">
                                    <?php echo htmlspecialchars($productRow['category_name'] ?? 'Uncategorised'); ?>
                                </td>
                                <td class="px-4 py-3">
                                    <div class="flex flex-col">
                                        <span class="font-semibold text-gray-900"><?php echo formatCurrency($productRow['sale_price'] ?? $productRow['price']); ?></span>
                                        <?php if ($productRow['sale_price']): ?>
                                            <span class="text-xs text-gray-500 line-through"><?php echo formatCurrency($productRow['price']); ?></span>
                                        <?php endif; ?>
                                    </div>
                                </td>
                                <td class="px-4 py-3 text-gray-600">
                                    <?php echo intval($productRow['stock_quantity']); ?>
                                </td>
                                <td class="px-4 py-3">
                                    <span class="badge inline-block <?php echo $productRow['status'] === 'active' ? 'bg-green-100 text-green-800' : 'bg-gray-200 text-gray-700'; ?>">
                                        <?php echo ucfirst($productRow['status']); ?>
                                    </span>
                                </td>
                                <td class="px-4 py-3">
                                    <div class="flex gap-2 flex-wrap">
                                        <?php if ($productRow['featured']): ?>
                                            <span class="badge badge-featured">Featured</span>
                                        <?php endif; ?>
                                        <?php if ($productRow['new_arrival']): ?>
                                            <span class="badge badge-new">New</span>
                                        <?php endif; ?>
                                    </div>
                                </td>
                                <td class="px-4 py-3 text-right">
                                    <div class="flex items-center justify-end gap-2">
                                        <a href="<?php echo SITE_URL; ?>/admin/products.php?edit=<?php echo $productRow['id']; ?>" class="btn-outline text-xs">
                                            <i class="fas fa-edit mr-1"></i>Edit
                                        </a>
                                        <form method="POST" onsubmit="return confirm('Delete this product?');">
                                            <input type="hidden" name="action" value="delete">
                                            <input type="hidden" name="product_id" value="<?php echo $productRow['id']; ?>">
                                            <button type="submit" class="btn-outline text-xs text-red-600 border-red-300 hover:bg-red-50">
                                                <i class="fas fa-trash mr-1"></i>Delete
                                            </button>
                                        </form>
                                    </div>
                                </td>
                            </tr>
                        <?php endwhile; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>

<?php
$productsResult->free();
closeDBConnection($conn);
require_once __DIR__ . '/../includes/footer.php';
?>
