<?php
require_once __DIR__ . '/../config/config.php';

if (!isLoggedIn()) {
    $_SESSION['admin_redirect'] = true;
    redirect(SITE_URL . '/login.php?redirect=admin');
} elseif (!isAdmin()) {
    redirect(SITE_URL . '/index.php');
}

$activePage = 'hero-banners';
$pageTitle = 'Hero Banners | ' . SITE_NAME;

$conn = getDBConnection();
$errors = [];
$success = '';

// Check for session success message
if (isset($_SESSION['banner_success'])) {
    $success = $_SESSION['banner_success'];
    unset($_SESSION['banner_success']);
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';

    if ($action === 'create' || $action === 'update') {
        $id = isset($_POST['banner_id']) ? intval($_POST['banner_id']) : 0;
        $title = sanitize($_POST['title'] ?? '');
        $subtitle = sanitize($_POST['subtitle'] ?? '');
        $buttonText = sanitize($_POST['button_text'] ?? '');
        $buttonLink = sanitize($_POST['button_link'] ?? '');
        $imageUrl = sanitize($_POST['image_url'] ?? '');
        $secondaryImageUrl = sanitize($_POST['secondary_image_url'] ?? '');
        $bgPositionDesktop = sanitize($_POST['bg_position_desktop'] ?? 'center');
        $bgPositionMobile = sanitize($_POST['bg_position_mobile'] ?? 'center');
        $isActive = isset($_POST['is_active']) ? 1 : 0;
        $displayOrder = intval($_POST['display_order'] ?? 0);

        if (empty($title)) {
            $errors[] = 'Title is required';
        }

        if (empty($errors)) {
            if ($action === 'create') {
                $stmt = $conn->prepare("INSERT INTO hero_banners (title, subtitle, button_text, button_link, image_url, secondary_image_url, bg_position_desktop, bg_position_mobile, is_active, display_order) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
                $stmt->bind_param("ssssssssii", $title, $subtitle, $buttonText, $buttonLink, $imageUrl, $secondaryImageUrl, $bgPositionDesktop, $bgPositionMobile, $isActive, $displayOrder);
                
                if ($stmt->execute()) {
                    $stmt->close();
                    $_SESSION['banner_success'] = 'Hero banner created successfully';
                    redirect(SITE_URL . '/admin/hero-banners.php');
                } else {
                    $errors[] = 'Failed to create hero banner';
                    $stmt->close();
                }
            } else {
                $stmt = $conn->prepare("UPDATE hero_banners SET title = ?, subtitle = ?, button_text = ?, button_link = ?, image_url = ?, secondary_image_url = ?, bg_position_desktop = ?, bg_position_mobile = ?, is_active = ?, display_order = ? WHERE id = ?");
                if (!$stmt) {
                    $errors[] = 'Prepare failed: ' . $conn->error;
                } else {
                    $stmt->bind_param("ssssssssiii", $title, $subtitle, $buttonText, $buttonLink, $imageUrl, $secondaryImageUrl, $bgPositionDesktop, $bgPositionMobile, $isActive, $displayOrder, $id);
                    
                    if ($stmt->execute()) {
                        $stmt->close();
                        $_SESSION['banner_success'] = 'Hero banner updated successfully';
                        redirect(SITE_URL . '/admin/hero-banners.php');
                    } else {
                        $errors[] = 'Failed to update hero banner: ' . $stmt->error;
                        $stmt->close();
                    }
                }
            }
        }
    } elseif ($action === 'delete') {
        $bannerId = intval($_POST['banner_id'] ?? 0);
        if ($bannerId) {
            $stmt = $conn->prepare("DELETE FROM hero_banners WHERE id = ?");
            $stmt->bind_param("i", $bannerId);
            if ($stmt->execute()) {
                $stmt->close();
                $_SESSION['banner_success'] = 'Hero banner deleted successfully';
                redirect(SITE_URL . '/admin/hero-banners.php');
            } else {
                $errors[] = 'Failed to delete hero banner';
                $stmt->close();
            }
        }
    }
}

// Get hero banner to edit
$editBanner = null;
if (isset($_GET['edit'])) {
    $editId = intval($_GET['edit']);
    $stmt = $conn->prepare("SELECT * FROM hero_banners WHERE id = ?");
    $stmt->bind_param("i", $editId);
    $stmt->execute();
    $result = $stmt->get_result();
    $editBanner = $result->fetch_assoc();
    $stmt->close();
}

// Get all hero banners
$bannersResult = $conn->query("SELECT * FROM hero_banners ORDER BY display_order ASC, created_at DESC");

require_once __DIR__ . '/../includes/header.php';
require_once __DIR__ . '/admin_nav.php';
?>

<div class="container mx-auto px-4 py-8">
    <div class="flex justify-between items-center mb-8">
        <h1 class="text-3xl font-display font-bold text-gray-900">Hero Banners</h1>
        <?php if (!$editBanner): ?>
            <button onclick="document.getElementById('createForm').scrollIntoView({behavior: 'smooth'})" class="btn-primary">
                <i class="fas fa-plus mr-2"></i>Add New Banner
            </button>
        <?php endif; ?>
    </div>

    <?php if ($success): ?>
        <div class="bg-green-50 border border-green-200 text-green-700 px-4 py-3 rounded-lg mb-6">
            <i class="fas fa-check-circle mr-2"></i><?php echo $success; ?>
        </div>
    <?php endif; ?>

    <?php if ($errors): ?>
        <div class="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-lg mb-6">
            <i class="fas fa-exclamation-circle mr-2"></i>
            <ul class="list-disc list-inside">
                <?php foreach ($errors as $error): ?>
                    <li><?php echo $error; ?></li>
                <?php endforeach; ?>
            </ul>
        </div>
    <?php endif; ?>

    <!-- Banner Form -->
    <div class="<?php echo $editBanner ? 'bg-primary-50 border-2 border-primary-300' : 'bg-white'; ?> rounded-xl shadow-md p-8 mb-8" id="createForm">
        <?php if ($editBanner): ?>
            <div class="bg-primary-600 text-white px-4 py-3 rounded-lg mb-6 flex items-center justify-between">
                <div class="flex items-center">
                    <i class="fas fa-edit text-xl mr-3"></i>
                    <div>
                        <h3 class="font-bold text-lg">Editing Banner</h3>
                        <p class="text-sm text-primary-100">Make changes to "<?php echo htmlspecialchars($editBanner['title']); ?>"</p>
                    </div>
                </div>
                <a href="<?php echo SITE_URL; ?>/admin/hero-banners.php" class="bg-white text-primary-600 px-4 py-2 rounded-lg text-sm font-semibold hover:bg-primary-50 transition">
                    <i class="fas fa-times mr-1"></i>Cancel Edit
                </a>
            </div>
        <?php endif; ?>
        
        <h2 class="text-2xl font-display font-bold text-gray-900 mb-6">
            <?php echo $editBanner ? 'Edit Hero Banner Details' : 'Create New Hero Banner'; ?>
        </h2>
        
        <form method="POST">
            <input type="hidden" name="action" value="<?php echo $editBanner ? 'update' : 'create'; ?>">
            <?php if ($editBanner): ?>
                <input type="hidden" name="banner_id" value="<?php echo $editBanner['id']; ?>">
            <?php endif; ?>

            <div class="grid md:grid-cols-2 gap-6 mb-6">
                <div>
                    <label class="block text-gray-700 font-semibold mb-2">Title *</label>
                    <input type="text" name="title" required class="input-field" 
                           value="<?php echo htmlspecialchars($editBanner['title'] ?? ''); ?>"
                           placeholder="e.g., Discover Natural Beauty">
                </div>

                <div>
                    <label class="block text-gray-700 font-semibold mb-2">Display Order</label>
                    <input type="number" name="display_order" class="input-field" 
                           value="<?php echo $editBanner['display_order'] ?? 0; ?>"
                           placeholder="0">
                </div>
            </div>

            <div class="mb-6">
                <label class="block text-gray-700 font-semibold mb-2">Subtitle</label>
                <textarea name="subtitle" rows="3" class="input-field" 
                          placeholder="Short description or tagline"><?php echo htmlspecialchars($editBanner['subtitle'] ?? ''); ?></textarea>
            </div>

            <div class="grid md:grid-cols-2 gap-6 mb-6">
                <div>
                    <label class="block text-gray-700 font-semibold mb-2">Button Text</label>
                    <input type="text" name="button_text" class="input-field" 
                           value="<?php echo htmlspecialchars($editBanner['button_text'] ?? ''); ?>"
                           placeholder="e.g., Shop Now">
                </div>

                <div>
                    <label class="block text-gray-700 font-semibold mb-2">Button Link</label>
                    <input type="text" name="button_link" class="input-field" 
                           value="<?php echo htmlspecialchars($editBanner['button_link'] ?? ''); ?>"
                           placeholder="e.g., /products.php or <?php echo SITE_URL; ?>/products.php">
                </div>
            </div>

            <div class="mb-6">
                <label class="block text-gray-700 font-semibold mb-2">Background Image URL</label>
                <input type="text" name="image_url" class="input-field" 
                       value="<?php echo htmlspecialchars($editBanner['image_url'] ?? ''); ?>"
                       placeholder="https://images.unsplash.com/...">
                <p class="text-sm text-gray-500 mt-1">Recommended size: 1920x800px - Full background image for the hero section</p>
            </div>

            <div class="mb-6">
                <label class="block text-gray-700 font-semibold mb-2">Right Side Image URL (Desktop Only)</label>
                <input type="text" name="secondary_image_url" class="input-field" 
                       value="<?php echo htmlspecialchars($editBanner['secondary_image_url'] ?? ''); ?>"
                       placeholder="https://images.unsplash.com/...">
                <p class="text-sm text-gray-500 mt-1">Recommended size: 800x800px - Product or lifestyle image shown on right side</p>
            </div>

            <div class="grid md:grid-cols-2 gap-6 mb-6">
                <div>
                    <label class="block text-gray-700 font-semibold mb-2">Background Position (Desktop)</label>
                    <select name="bg_position_desktop" class="input-field">
                        <option value="left" <?php echo ($editBanner['bg_position_desktop'] ?? 'center') === 'left' ? 'selected' : ''; ?>>Left</option>
                        <option value="center" <?php echo ($editBanner['bg_position_desktop'] ?? 'center') === 'center' ? 'selected' : ''; ?>>Center</option>
                        <option value="right" <?php echo ($editBanner['bg_position_desktop'] ?? 'center') === 'right' ? 'selected' : ''; ?>>Right</option>
                        <option value="top" <?php echo ($editBanner['bg_position_desktop'] ?? 'center') === 'top' ? 'selected' : ''; ?>>Top</option>
                        <option value="bottom" <?php echo ($editBanner['bg_position_desktop'] ?? 'center') === 'bottom' ? 'selected' : ''; ?>>Bottom</option>
                        <option value="left top" <?php echo ($editBanner['bg_position_desktop'] ?? 'center') === 'left top' ? 'selected' : ''; ?>>Left Top</option>
                        <option value="left bottom" <?php echo ($editBanner['bg_position_desktop'] ?? 'center') === 'left bottom' ? 'selected' : ''; ?>>Left Bottom</option>
                        <option value="right top" <?php echo ($editBanner['bg_position_desktop'] ?? 'center') === 'right top' ? 'selected' : ''; ?>>Right Top</option>
                        <option value="right bottom" <?php echo ($editBanner['bg_position_desktop'] ?? 'center') === 'right bottom' ? 'selected' : ''; ?>>Right Bottom</option>
                    </select>
                    <p class="text-sm text-gray-500 mt-1">How the image is positioned on desktop</p>
                </div>

                <div>
                    <label class="block text-gray-700 font-semibold mb-2">Background Position (Mobile)</label>
                    <select name="bg_position_mobile" class="input-field">
                        <option value="left" <?php echo ($editBanner['bg_position_mobile'] ?? 'center') === 'left' ? 'selected' : ''; ?>>Left</option>
                        <option value="center" <?php echo ($editBanner['bg_position_mobile'] ?? 'center') === 'center' ? 'selected' : ''; ?>>Center</option>
                        <option value="right" <?php echo ($editBanner['bg_position_mobile'] ?? 'center') === 'right' ? 'selected' : ''; ?>>Right</option>
                        <option value="top" <?php echo ($editBanner['bg_position_mobile'] ?? 'center') === 'top' ? 'selected' : ''; ?>>Top</option>
                        <option value="bottom" <?php echo ($editBanner['bg_position_mobile'] ?? 'center') === 'bottom' ? 'selected' : ''; ?>>Bottom</option>
                        <option value="left top" <?php echo ($editBanner['bg_position_mobile'] ?? 'center') === 'left top' ? 'selected' : ''; ?>>Left Top</option>
                        <option value="left bottom" <?php echo ($editBanner['bg_position_mobile'] ?? 'center') === 'left bottom' ? 'selected' : ''; ?>>Left Bottom</option>
                        <option value="right top" <?php echo ($editBanner['bg_position_mobile'] ?? 'center') === 'right top' ? 'selected' : ''; ?>>Right Top</option>
                        <option value="right bottom" <?php echo ($editBanner['bg_position_mobile'] ?? 'center') === 'right bottom' ? 'selected' : ''; ?>>Right Bottom</option>
                    </select>
                    <p class="text-sm text-gray-500 mt-1">How the image is positioned on mobile</p>
                </div>
            </div>

            <div class="mb-6">
                <label class="flex items-center">
                    <input type="checkbox" name="is_active" value="1" 
                           <?php echo ($editBanner['is_active'] ?? true) ? 'checked' : ''; ?>
                           class="w-4 h-4 text-primary-600 border-gray-300 rounded focus:ring-primary-500">
                    <span class="ml-2 text-gray-700 font-semibold">Active (Show on homepage)</span>
                </label>
            </div>

            <div class="flex gap-4">
                <button type="submit" class="btn-primary">
                    <i class="fas fa-save mr-2"></i><?php echo $editBanner ? 'Update Banner' : 'Create Banner'; ?>
                </button>
                <?php if ($editBanner): ?>
                    <a href="<?php echo SITE_URL; ?>/admin/hero-banners.php" class="btn-outline">
                        Cancel
                    </a>
                <?php endif; ?>
            </div>
        </form>
    </div>

    <!-- Banners List -->
    <div class="bg-white rounded-xl shadow-md overflow-hidden">
        <div class="px-6 py-4 border-b border-gray-200">
            <h3 class="text-xl font-display font-bold text-gray-900">All Hero Banners</h3>
        </div>

        <div class="overflow-x-auto">
            <table class="w-full text-sm">
                <thead class="bg-gray-50">
                    <tr>
                        <th class="px-4 py-3 text-left font-semibold text-gray-600">Banner</th>
                        <th class="px-4 py-3 text-left font-semibold text-gray-600">Details</th>
                        <th class="px-4 py-3 text-left font-semibold text-gray-600">Order</th>
                        <th class="px-4 py-3 text-left font-semibold text-gray-600">Status</th>
                        <th class="px-4 py-3 text-right font-semibold text-gray-600">Actions</th>
                    </tr>
                </thead>
                <tbody class="divide-y divide-gray-100">
                    <?php while ($banner = $bannersResult->fetch_assoc()): ?>
                        <tr class="hover:bg-gray-50">
                            <td class="px-4 py-3">
                                <div class="flex items-center gap-3">
                                    <?php if ($banner['image_url']): ?>
                                        <div class="w-24 h-12 rounded-lg overflow-hidden bg-gray-100">
                                            <img src="<?php echo htmlspecialchars($banner['image_url']); ?>" 
                                                 alt="Banner preview" class="w-full h-full object-cover">
                                        </div>
                                    <?php endif; ?>
                                    <div>
                                        <p class="font-semibold text-gray-900"><?php echo htmlspecialchars($banner['title']); ?></p>
                                        <?php if ($banner['subtitle']): ?>
                                            <p class="text-xs text-gray-500 line-clamp-1"><?php echo htmlspecialchars($banner['subtitle']); ?></p>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </td>
                            <td class="px-4 py-3 text-gray-600">
                                <?php if ($banner['button_text']): ?>
                                    <p class="text-xs"><strong>Button:</strong> <?php echo htmlspecialchars($banner['button_text']); ?></p>
                                <?php endif; ?>
                                <?php if ($banner['button_link']): ?>
                                    <p class="text-xs text-gray-500"><?php echo htmlspecialchars($banner['button_link']); ?></p>
                                <?php endif; ?>
                            </td>
                            <td class="px-4 py-3 text-gray-600">
                                <?php echo intval($banner['display_order']); ?>
                            </td>
                            <td class="px-4 py-3">
                                <span class="badge inline-block <?php echo $banner['is_active'] ? 'bg-green-100 text-green-800' : 'bg-gray-200 text-gray-700'; ?>">
                                    <?php echo $banner['is_active'] ? 'Active' : 'Inactive'; ?>
                                </span>
                            </td>
                            <td class="px-4 py-3 text-right">
                                <div class="flex items-center justify-end gap-2">
                                    <a href="<?php echo SITE_URL; ?>/admin/hero-banners.php?edit=<?php echo $banner['id']; ?>" 
                                       class="btn-outline text-xs">
                                        <i class="fas fa-edit mr-1"></i>Edit
                                    </a>
                                    <form method="POST" onsubmit="return confirm('Delete this banner?');">
                                        <input type="hidden" name="action" value="delete">
                                        <input type="hidden" name="banner_id" value="<?php echo $banner['id']; ?>">
                                        <button type="submit" class="btn-outline text-xs text-red-600 border-red-300 hover:bg-red-50">
                                            <i class="fas fa-trash mr-1"></i>Delete
                                        </button>
                                    </form>
                                </div>
                            </td>
                        </tr>
                    <?php endwhile; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<?php if ($editBanner): ?>
<script>
    // Auto-scroll to form when editing
    (function() {
        let scrolled = false;
        document.addEventListener('DOMContentLoaded', function() {
            if (scrolled) return;
            scrolled = true;
            
            const form = document.getElementById('createForm');
            if (form) {
                setTimeout(function() {
                    form.scrollIntoView({ behavior: 'smooth', block: 'start' });
                    form.style.animation = 'pulse 0.5s ease-in-out';
                }, 100);
            }
        });
    })();
</script>
<style>
    @keyframes pulse {
        0%, 100% { transform: scale(1); }
        50% { transform: scale(1.01); }
    }
</style>
<?php endif; ?>

<?php
$bannersResult->free();
closeDBConnection($conn);
require_once __DIR__ . '/../includes/footer.php';
?>
