<?php
require_once 'config/config.php';
$pageTitle = "Shopping Cart | " . SITE_NAME;
require_once 'includes/header.php';

$conn = getDBConnection();

// Get cart items
if (isLoggedIn()) {
    $userId = getCurrentUserId();
    $stmt = $conn->prepare("
        SELECT c.id, c.quantity, p.id as product_id, p.name, p.slug, p.price, p.sale_price, p.stock_quantity, pi.image_url
        FROM cart c
        JOIN products p ON c.product_id = p.id
        LEFT JOIN product_images pi ON p.id = pi.product_id AND pi.is_primary = 1
        WHERE c.user_id = ? AND p.status = 'active'
    ");
    $stmt->bind_param("i", $userId);
} else {
    $sessionId = getCartSessionId();
    $stmt = $conn->prepare("
        SELECT c.id, c.quantity, p.id as product_id, p.name, p.slug, p.price, p.sale_price, p.stock_quantity, pi.image_url
        FROM cart c
        JOIN products p ON c.product_id = p.id
        LEFT JOIN product_images pi ON p.id = pi.product_id AND pi.is_primary = 1
        WHERE c.session_id = ? AND p.status = 'active'
    ");
    $stmt->bind_param("s", $sessionId);
}

$stmt->execute();
$cartItems = $stmt->get_result();
$stmt->close();

$subtotal = 0;
$cartItemsArray = [];
while ($item = $cartItems->fetch_assoc()) {
    $itemPrice = $item['sale_price'] ?? $item['price'];
    $item['item_total'] = $itemPrice * $item['quantity'];
    $subtotal += $item['item_total'];
    $cartItemsArray[] = $item;
}

$shippingCost = $subtotal >= FREE_SHIPPING_THRESHOLD ? 0 : 150;
$total = $subtotal + $shippingCost;

closeDBConnection($conn);
?>

<section class="py-12 min-h-screen">
    <div class="container mx-auto px-4">
        <h1 class="text-4xl md:text-5xl font-display font-bold text-gray-900 mb-8">Shopping Cart</h1>

        <?php if (count($cartItemsArray) > 0): ?>
            <div class="grid lg:grid-cols-3 gap-8">
                <!-- Cart Items -->
                <div class="lg:col-span-2 space-y-4">
                    <?php foreach ($cartItemsArray as $item): 
                        $itemImage = getProductImage($item['image_url'] ?? null, $item['product_id']);
                        $itemImageIsPlaceholder = isPlaceholderImage($itemImage);
                    ?>
                        <div class="bg-white rounded-xl shadow-md p-6 flex flex-col md:flex-row gap-6">
                            <a href="<?php echo SITE_URL; ?>/product.php?slug=<?php echo $item['slug']; ?>" 
                               class="flex-shrink-0">
                                <div class="relative w-32 h-32 bg-gray-100 rounded-lg overflow-hidden">
                                    <img src="<?php echo $itemImage; ?>" 
                                         alt="<?php echo htmlspecialchars($item['name']); ?>"
                                         class="w-full h-full object-cover">
                                    <?php if ($itemImageIsPlaceholder): ?>
                                        <div class="absolute inset-0 bg-gray-900/40 flex items-center justify-center">
                                            <span class="text-white text-[10px] font-semibold tracking-wide uppercase text-center px-2">Image coming soon</span>
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </a>

                            <div class="flex-1">
                                <a href="<?php echo SITE_URL; ?>/product.php?slug=<?php echo $item['slug']; ?>">
                                    <h3 class="font-display font-bold text-xl text-gray-900 mb-2 hover:text-primary-600 transition">
                                        <?php echo htmlspecialchars($item['name']); ?>
                                    </h3>
                                </a>
                                
                                <div class="flex items-center space-x-4 mb-4">
                                    <?php if ($item['sale_price']): ?>
                                        <span class="text-lg font-bold text-primary-600">
                                            <?php echo formatCurrency($item['sale_price']); ?>
                                        </span>
                                        <span class="text-sm text-gray-500 line-through">
                                            <?php echo formatCurrency($item['price']); ?>
                                        </span>
                                    <?php else: ?>
                                        <span class="text-lg font-bold text-gray-900">
                                            <?php echo formatCurrency($item['price']); ?>
                                        </span>
                                    <?php endif; ?>
                                </div>

                                <div class="flex items-center justify-between">
                                    <div class="flex items-center border border-gray-300 rounded-lg">
                                        <button onclick="updateCartQty(<?php echo $item['id']; ?>, <?php echo $item['quantity'] - 1; ?>, <?php echo $item['product_id']; ?>)" 
                                                class="px-3 py-2 hover:bg-gray-100 transition">
                                            <i class="fas fa-minus text-sm"></i>
                                        </button>
                                        <input type="number" value="<?php echo $item['quantity']; ?>" readonly
                                               class="w-12 text-center border-x border-gray-300 py-2 focus:outline-none">
                                        <button onclick="updateCartQty(<?php echo $item['id']; ?>, <?php echo $item['quantity'] + 1; ?>, <?php echo $item['product_id']; ?>)" 
                                                class="px-3 py-2 hover:bg-gray-100 transition">
                                            <i class="fas fa-plus text-sm"></i>
                                        </button>
                                    </div>

                                    <div class="flex items-center space-x-4">
                                        <span class="font-bold text-xl text-gray-900">
                                            <?php echo formatCurrency($item['item_total']); ?>
                                        </span>
                                        <button onclick="removeFromCart(<?php echo $item['id']; ?>)" 
                                                class="text-red-500 hover:text-red-700 transition">
                                            <i class="fas fa-trash-alt"></i>
                                        </button>
                                    </div>
                                </div>

                                <?php if ($item['stock_quantity'] < $item['quantity']): ?>
                                    <div class="mt-2 text-sm text-red-600">
                                        <i class="fas fa-exclamation-circle mr-1"></i>
                                        Only <?php echo $item['stock_quantity']; ?> left in stock
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>

                <!-- Order Summary -->
                <div class="lg:col-span-1">
                    <div class="bg-white rounded-xl shadow-md p-6 sticky top-24">
                        <h2 class="text-2xl font-display font-bold text-gray-900 mb-6">Order Summary</h2>
                        
                        <div class="space-y-3 mb-6">
                            <div class="flex justify-between text-gray-600">
                                <span>Subtotal</span>
                                <span class="font-semibold"><?php echo formatCurrency($subtotal); ?></span>
                            </div>
                            <div class="flex justify-between text-gray-600">
                                <span>Shipping</span>
                                <?php if ($shippingCost > 0): ?>
                                    <span class="font-semibold"><?php echo formatCurrency($shippingCost); ?></span>
                                <?php else: ?>
                                    <span class="font-semibold text-green-600">FREE</span>
                                <?php endif; ?>
                            </div>
                            
                            <?php if ($subtotal < FREE_SHIPPING_THRESHOLD && $subtotal > 0): ?>
                                <div class="bg-primary-50 text-primary-700 p-3 rounded-lg text-sm">
                                    <i class="fas fa-info-circle mr-1"></i>
                                    Add <?php echo formatCurrency(FREE_SHIPPING_THRESHOLD - $subtotal); ?> more for free shipping
                                </div>
                            <?php endif; ?>
                            
                            <div class="border-t border-gray-200 pt-3 flex justify-between text-lg font-bold">
                                <span>Total</span>
                                <span class="text-primary-600"><?php echo formatCurrency($total); ?></span>
                            </div>
                        </div>

                        <a href="<?php echo SITE_URL; ?>/checkout.php" class="btn-primary w-full text-lg mb-3">
                            <i class="fas fa-lock mr-2"></i>Proceed to Checkout
                        </a>
                        <a href="<?php echo SITE_URL; ?>/products.php" class="btn-outline w-full">
                            <i class="fas fa-arrow-left mr-2"></i>Continue Shopping
                        </a>

                        <div class="mt-6 space-y-3 text-sm text-gray-600">
                            <div class="flex items-center">
                                <i class="fas fa-shield-alt text-primary-600 mr-3"></i>
                                <span>Secure checkout</span>
                            </div>
                            <div class="flex items-center">
                                <i class="fas fa-undo text-primary-600 mr-3"></i>
                                <span>30-day returns</span>
                            </div>
                            <div class="flex items-center">
                                <i class="fas fa-truck text-primary-600 mr-3"></i>
                                <span>Fast delivery</span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        <?php else: ?>
            <div class="text-center py-20">
                <div class="text-gray-400 text-6xl mb-4">
                    <i class="fas fa-shopping-bag"></i>
                </div>
                <h2 class="text-3xl font-display font-bold text-gray-900 mb-4">Your Cart is Empty</h2>
                <p class="text-gray-600 mb-8">Looks like you haven't added any products yet</p>
                <a href="<?php echo SITE_URL; ?>/products.php" class="btn-primary">
                    <i class="fas fa-arrow-left mr-2"></i>Start Shopping
                </a>
            </div>
        <?php endif; ?>
    </div>
</section>

<script>
function updateCartQty(cartId, newQty, productId) {
    if (newQty < 1) {
        removeFromCart(cartId);
        return;
    }

    fetch('<?php echo SITE_URL; ?>/api/update-cart.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({ cart_id: cartId, quantity: newQty, product_id: productId })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            location.reload();
        } else {
            alert(data.message || 'Failed to update cart');
        }
    })
    .catch(error => {
        alert('An error occurred. Please try again.');
    });
}

function removeFromCart(cartId) {
    if (!confirm('Remove this item from cart?')) return;

    fetch('<?php echo SITE_URL; ?>/api/remove-from-cart.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({ cart_id: cartId })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            location.reload();
        } else {
            alert(data.message || 'Failed to remove item');
        }
    })
    .catch(error => {
        alert('An error occurred. Please try again.');
    });
}
</script>

<?php require_once 'includes/footer.php'; ?>
