<?php
require_once __DIR__ . '/../config/config.php';

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method']);
    exit;
}

$input = json_decode(file_get_contents('php://input'), true);
$productId = isset($input['product_id']) ? intval($input['product_id']) : 0;
$quantity = isset($input['quantity']) ? intval($input['quantity']) : 1;

if ($productId <= 0 || $quantity <= 0) {
    echo json_encode(['success' => false, 'message' => 'Invalid product or quantity']);
    exit;
}

$conn = getDBConnection();

// Check if product exists and is in stock
$stmt = $conn->prepare("SELECT id, name, stock_quantity FROM products WHERE id = ? AND status = 'active'");
$stmt->bind_param("i", $productId);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows == 0) {
    $stmt->close();
    closeDBConnection($conn);
    echo json_encode(['success' => false, 'message' => 'Product not found']);
    exit;
}

$product = $result->fetch_assoc();
$stmt->close();

if ($product['stock_quantity'] < $quantity) {
    closeDBConnection($conn);
    echo json_encode(['success' => false, 'message' => 'Insufficient stock']);
    exit;
}

// Add to cart
if (isLoggedIn()) {
    $userId = getCurrentUserId();
    
    // Check if product already in cart
    $checkStmt = $conn->prepare("SELECT id, quantity FROM cart WHERE user_id = ? AND product_id = ?");
    $checkStmt->bind_param("ii", $userId, $productId);
    $checkStmt->execute();
    $checkResult = $checkStmt->get_result();
    
    if ($checkResult->num_rows > 0) {
        // Update quantity
        $cartItem = $checkResult->fetch_assoc();
        $newQuantity = $cartItem['quantity'] + $quantity;
        
        $updateStmt = $conn->prepare("UPDATE cart SET quantity = ? WHERE id = ?");
        $updateStmt->bind_param("ii", $newQuantity, $cartItem['id']);
        $updateStmt->execute();
        $updateStmt->close();
    } else {
        // Insert new cart item
        $insertStmt = $conn->prepare("INSERT INTO cart (user_id, product_id, quantity) VALUES (?, ?, ?)");
        $insertStmt->bind_param("iii", $userId, $productId, $quantity);
        $insertStmt->execute();
        $insertStmt->close();
    }
    $checkStmt->close();
} else {
    $sessionId = getCartSessionId();
    
    // Check if product already in cart
    $checkStmt = $conn->prepare("SELECT id, quantity FROM cart WHERE session_id = ? AND product_id = ?");
    $checkStmt->bind_param("si", $sessionId, $productId);
    $checkStmt->execute();
    $checkResult = $checkStmt->get_result();
    
    if ($checkResult->num_rows > 0) {
        // Update quantity
        $cartItem = $checkResult->fetch_assoc();
        $newQuantity = $cartItem['quantity'] + $quantity;
        
        $updateStmt = $conn->prepare("UPDATE cart SET quantity = ? WHERE id = ?");
        $updateStmt->bind_param("ii", $newQuantity, $cartItem['id']);
        $updateStmt->execute();
        $updateStmt->close();
    } else {
        // Insert new cart item
        $insertStmt = $conn->prepare("INSERT INTO cart (session_id, product_id, quantity) VALUES (?, ?, ?)");
        $insertStmt->bind_param("sii", $sessionId, $productId, $quantity);
        $insertStmt->execute();
        $insertStmt->close();
    }
    $checkStmt->close();
}

closeDBConnection($conn);
echo json_encode(['success' => true, 'message' => 'Product added to cart']);
