# 🚀 Production Deployment Guide

## ✅ Recent Updates (Auto-Configuration)

**Good News!** Your site now automatically detects its URL and paths, making deployment much easier:

- ✅ **SITE_URL** is now automatically detected (no manual changes needed)
- ✅ **Asset paths** work correctly from any directory or subdomain
- ✅ **HTTPS** is automatically detected
- ✅ **.htaccess** uses relative paths (works anywhere)

### What This Means:
- Deploy to any domain without changing `SITE_URL`
- Works in subdirectories (e.g., `yourdomain.com/shop`)
- Automatic HTTPS detection
- No more hardcoded localhost references

## Pre-Deployment Checklist

### 1. Update Configuration Files

#### A. Update `config/database.php`
```php
// Change these for production:
define('DB_HOST', 'your_production_db_host');
define('DB_USER', 'your_production_db_user');
define('DB_PASS', 'your_production_db_password');
define('DB_NAME', 'your_production_db_name');
```

**Note:** `SITE_URL` in `config/config.php` is now auto-detected and doesn't need manual changes!

#### B. `.htaccess` Configuration

Your `.htaccess` file is now production-ready with relative paths! 

**Optional:** To force HTTPS in production, uncomment these lines in `.htaccess`:
```apache
# Line 48-52: Force HTTPS in production
<IfModule mod_rewrite.c>
    RewriteCond %{HTTPS} off
    RewriteCond %{HTTP_HOST} !^localhost [NC]
    RewriteRule ^(.*)$ https://%{HTTP_HOST}%{REQUEST_URI} [L,R=301]
</IfModule>
```

**No other .htaccess changes needed!** It now uses relative paths that work everywhere.

### 2. Database Setup

#### A. Export Your Database
```bash
# From your local XAMPP
/Applications/XAMPP/xamppfiles/bin/mysqldump -u root skincare_shop > skincare_shop_export.sql
```

#### B. Import to Production Server
```bash
# On your production server
mysql -u your_db_user -p your_db_name < skincare_shop_export.sql
```

### 3. Upload Files

#### Files to Upload:
- All `.php` files
- `assets/` folder
- `includes/` folder
- `config/` folder (update first!)
- `admin/` folder
- `uploads/` folder (create if doesn't exist)
- `.htaccess` (production version)
- `api/` folder

#### Files to NEVER Upload:
- `database.sql`
- `.htaccess.local`
- `.htaccess.production`
- `DEPLOYMENT_GUIDE.md`
- Any test files

### 4. Set Folder Permissions

```bash
# On your production server
chmod 755 uploads/
chmod 755 uploads/products/
chmod 755 uploads/categories/
chmod 644 config/config.php
chmod 644 .htaccess
```

### 5. Test Your Site

#### A. Quick Configuration Check

Visit `https://yourdomain.com/check-config.php` to verify:
- ✅ CSS files are loading correctly
- ✅ Site URL is properly detected
- ✅ Database connection works
- ✅ Upload directories have correct permissions
- ✅ Server configuration is correct

**Important:** Delete or restrict access to `check-config.php` after testing!

#### B. Test These Pages:
- ✅ Homepage: `https://yourdomain.com`
- ✅ Products: `https://yourdomain.com/products.php`
- ✅ Single Product: `https://yourdomain.com/product.php?slug=any-product`
- ✅ Cart: `https://yourdomain.com/cart.php`
- ✅ Login: `https://yourdomain.com/login.php`
- ✅ Admin: `https://yourdomain.com/admin/`
- ✅ 404 Page: `https://yourdomain.com/nonexistent-page`

#### C. Test Admin Functions:
- Login with admin credentials
- Add/edit products
- Manage categories
- Update hero banner
- View orders

### 6. Security Checklist

#### Required:
- [ ] HTTPS/SSL certificate installed
- [ ] Production database credentials set
- [ ] `display_errors` set to Off in `.htaccess`
- [ ] Strong admin password
- [ ] Database user has minimum required permissions
- [ ] `.env` or sensitive files not accessible via URL
- [ ] `uploads/` folder secured (no PHP execution)

#### Recommended:
- [ ] Regular database backups scheduled
- [ ] Error logging enabled
- [ ] CDN for images (optional)
- [ ] Cloudflare or similar for DDoS protection

### 7. Common Issues & Solutions

#### Issue: 404 Page Not Working
**Solution:** Check `.htaccess` ErrorDocument paths match your domain structure

#### Issue: Images Not Loading
**Solution:** 
1. Check `uploads/` folder exists with correct permissions
2. Verify `SITE_URL` in `config/config.php` is correct
3. Check image paths in database

#### Issue: Database Connection Error
**Solution:** Verify database credentials in `config/config.php`

#### Issue: Admin Panel Not Accessible
**Solution:** 
1. Check you're using HTTPS if force HTTPS is enabled
2. Verify admin user exists in database
3. Check session is working (PHP session support)

### 8. Performance Optimization

#### Enable OpCache (if available):
```php
; In php.ini
opcache.enable=1
opcache.memory_consumption=128
opcache.max_accelerated_files=10000
```

#### Enable Gzip Compression:
Already configured in `.htaccess.production`

#### Browser Caching:
Already configured in `.htaccess.production`

### 9. Maintenance Mode (Optional)

Create `maintenance.php` and use `.htaccess` to redirect:
```apache
# Add to .htaccess for maintenance
RewriteCond %{REQUEST_URI} !^/maintenance\.php$
RewriteCond %{REMOTE_ADDR} !^123\.456\.789\.000$ # Your IP
RewriteRule ^(.*)$ /maintenance.php [R=302,L]
```

### 10. Post-Deployment

#### A. Monitor Error Logs
```bash
# Check PHP error log
tail -f /var/log/php_errors.log

# Check Apache error log
tail -f /var/log/apache2/error.log
```

#### B. Setup Monitoring
- Google Analytics
- Google Search Console
- Uptime monitoring (UptimeRobot, Pingdom, etc.)

## Quick Deploy Script

Create `deploy.sh` for quick updates:
```bash
#!/bin/bash
# Quick deployment script

# Backup database
mysqldump -u root skincare_shop > backup_$(date +%Y%m%d_%H%M%S).sql

# Upload files (adjust paths)
rsync -avz --exclude 'database.sql' --exclude '.git' \
  /path/to/local/skin/ user@yourserver:/path/to/production/

echo "Deployment complete!"
```

## Support

For issues, check:
1. PHP error logs
2. Apache error logs
3. Browser console (F12)
4. Database connection

---

**Remember:** Always backup before deploying updates!
