<?php
require_once 'config/config.php';
$pageTitle = "Shop All Products | " . SITE_NAME;

$conn = getDBConnection();

// Get filter parameters
$category = isset($_GET['category']) ? sanitize($_GET['category']) : '';
$search = isset($_GET['search']) ? sanitize($_GET['search']) : '';
$sort = isset($_GET['sort']) ? sanitize($_GET['sort']) : 'newest';
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$offset = ($page - 1) * PRODUCTS_PER_PAGE;

// Build query
$where = ["p.status = 'active'"];
$params = [];
$types = "";

if ($category) {
    $where[] = "c.slug = ?";
    $params[] = $category;
    $types .= "s";
    
    // Get category name
    $catStmt = $conn->prepare("SELECT name FROM categories WHERE slug = ?");
    $catStmt->bind_param("s", $category);
    $catStmt->execute();
    $catResult = $catStmt->get_result();
    $categoryName = $catResult->fetch_assoc()['name'] ?? '';
    $catStmt->close();
    $pageTitle = $categoryName . " | " . SITE_NAME;
}

if ($search) {
    $where[] = "(p.name LIKE ? OR p.description LIKE ? OR p.short_description LIKE ?)";
    $searchTerm = "%$search%";
    $params[] = $searchTerm;
    $params[] = $searchTerm;
    $params[] = $searchTerm;
    $types .= "sss";
    $pageTitle = "Search: $search | " . SITE_NAME;
}

$whereClause = implode(" AND ", $where);

// Order by
$orderBy = match($sort) {
    'price_low' => 'COALESCE(p.sale_price, p.price) ASC',
    'price_high' => 'COALESCE(p.sale_price, p.price) DESC',
    'name' => 'p.name ASC',
    default => 'p.created_at DESC',
};

// Count total products
$countQuery = "
    SELECT COUNT(DISTINCT p.id) as total 
    FROM products p 
    LEFT JOIN categories c ON p.category_id = c.id 
    WHERE $whereClause
";
$countStmt = $conn->prepare($countQuery);
if ($params) {
    $countStmt->bind_param($types, ...$params);
}
$countStmt->execute();
$totalProducts = $countStmt->get_result()->fetch_assoc()['total'];
$countStmt->close();
$totalPages = ceil($totalProducts / PRODUCTS_PER_PAGE);

// Get products
$query = "
    SELECT p.*, pi.image_url, c.name as category_name, c.slug as category_slug
    FROM products p 
    LEFT JOIN product_images pi ON p.id = pi.product_id AND pi.is_primary = 1
    LEFT JOIN categories c ON p.category_id = c.id
    WHERE $whereClause 
    ORDER BY $orderBy 
    LIMIT ? OFFSET ?
";

$stmt = $conn->prepare($query);
$params[] = PRODUCTS_PER_PAGE;
$params[] = $offset;
$types .= "ii";
$stmt->bind_param($types, ...$params);
$stmt->execute();
$products = $stmt->get_result();
$stmt->close();

// Get all categories for filter
$categories = $conn->query("SELECT * FROM categories ORDER BY display_order ASC");

require_once 'includes/header.php';
?>

<!-- Page Header -->
<section class="bg-gradient-to-r from-primary-50 to-primary-100 py-12">
    <div class="container mx-auto px-4">
        <h1 class="text-4xl md:text-5xl font-display font-bold text-gray-900 mb-4">
            <?php 
            if ($search) {
                echo "Search Results for: " . htmlspecialchars($search);
            } elseif ($category) {
                echo htmlspecialchars($categoryName ?? 'Products');
            } else {
                echo "All Products";
            }
            ?>
        </h1>
        <p class="text-gray-600">Showing <?php echo $products->num_rows; ?> of <?php echo $totalProducts; ?> products</p>
    </div>
</section>

<!-- Products Section -->
<section class="py-12">
    <div class="container mx-auto px-4">
        <div class="flex flex-col lg:flex-row gap-8">
            <!-- Filters Sidebar -->
            <aside class="lg:w-64 flex-shrink-0">
                <div class="bg-white rounded-xl shadow-md p-6 sticky top-24">
                    <h3 class="font-display font-bold text-xl mb-4">Filters</h3>
                    
                    <!-- Categories -->
                    <div class="mb-6">
                        <h4 class="font-semibold text-gray-900 mb-3">Categories</h4>
                        <div class="space-y-2">
                            <a href="<?php echo SITE_URL; ?>/products.php<?php echo $search ? '?search='.$search : ''; ?>" 
                               class="block py-2 px-3 rounded-lg <?php echo !$category ? 'bg-primary-100 text-primary-700' : 'text-gray-600 hover:bg-gray-100'; ?> transition">
                                All Products
                            </a>
                            <?php 
                            $categories->data_seek(0);
                            while ($cat = $categories->fetch_assoc()): 
                            ?>
                                <a href="<?php echo SITE_URL; ?>/products.php?category=<?php echo $cat['slug']; ?><?php echo $search ? '&search='.$search : ''; ?>" 
                                   class="block py-2 px-3 rounded-lg <?php echo $category == $cat['slug'] ? 'bg-primary-100 text-primary-700' : 'text-gray-600 hover:bg-gray-100'; ?> transition">
                                    <?php echo htmlspecialchars($cat['name']); ?>
                                </a>
                            <?php endwhile; ?>
                        </div>
                    </div>

                    <!-- Sort -->
                    <div>
                        <h4 class="font-semibold text-gray-900 mb-3">Sort By</h4>
                        <select onchange="window.location.href=this.value" class="input-field">
                            <?php
                            $baseUrl = SITE_URL . "/products.php?";
                            $params = [];
                            if ($category) $params[] = "category=$category";
                            if ($search) $params[] = "search=$search";
                            $paramStr = implode("&", $params);
                            $paramStr = $paramStr ? $paramStr . "&" : "";
                            ?>
                            <option value="<?php echo $baseUrl . $paramStr; ?>sort=newest" <?php echo $sort == 'newest' ? 'selected' : ''; ?>>Newest First</option>
                            <option value="<?php echo $baseUrl . $paramStr; ?>sort=price_low" <?php echo $sort == 'price_low' ? 'selected' : ''; ?>>Price: Low to High</option>
                            <option value="<?php echo $baseUrl . $paramStr; ?>sort=price_high" <?php echo $sort == 'price_high' ? 'selected' : ''; ?>>Price: High to Low</option>
                            <option value="<?php echo $baseUrl . $paramStr; ?>sort=name" <?php echo $sort == 'name' ? 'selected' : ''; ?>>Name: A to Z</option>
                        </select>
                    </div>
                </div>
            </aside>

            <!-- Products Grid -->
            <div class="flex-1">
                <?php if ($products->num_rows > 0): ?>
                    <div class="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-3 gap-6">
                        <?php while ($product = $products->fetch_assoc()): 
                            $productImage = getProductImage($product['image_url'], $product['id']);
                            $productImageIsPlaceholder = isPlaceholderImage($productImage);
                        ?>
                            <div class="card product-card group">
                                <a href="<?php echo SITE_URL; ?>/product.php?slug=<?php echo $product['slug']; ?>" class="block">
                                    <div class="relative overflow-hidden bg-gray-100 aspect-square">
                                        <img src="<?php echo $productImage; ?>" 
                                             alt="<?php echo htmlspecialchars($product['name']); ?>"
                                             class="w-full h-full object-cover">
                                        <?php if ($productImageIsPlaceholder): ?>
                                            <div class="absolute inset-0 bg-gray-900/40 flex items-center justify-center">
                                                <span class="text-white text-xs font-semibold tracking-wide uppercase">Image coming soon</span>
                                            </div>
                                        <?php endif; ?>

                                        <?php if ($product['new_arrival']): ?>
                                            <span class="badge badge-new absolute top-2 left-2">New</span>
                                        <?php endif; ?>
                                        <?php if ($product['sale_price']): ?>
                                            <span class="badge badge-sale absolute top-2 right-2">Sale</span>
                                        <?php endif; ?>
                                        <?php if ($product['featured']): ?>
                                            <span class="badge badge-featured absolute bottom-2 left-2">Featured</span>
                                        <?php endif; ?>
                                    </div>
                                </a>
                                <div class="p-4">
                                    <?php if ($product['category_name']): ?>
                                        <p class="text-xs text-primary-600 font-medium mb-1">
                                            <?php echo htmlspecialchars($product['category_name']); ?>
                                        </p>
                                    <?php endif; ?>
                                    <a href="<?php echo SITE_URL; ?>/product.php?slug=<?php echo $product['slug']; ?>">
                                        <h3 class="font-semibold text-gray-900 mb-2 line-clamp-2 group-hover:text-primary-600 transition">
                                            <?php echo htmlspecialchars($product['name']); ?>
                                        </h3>
                                    </a>
                                    <div class="mb-3">
                                        <?php if ($product['sale_price']): ?>
                                            <span class="text-lg font-bold text-primary-600">
                                                <?php echo formatCurrency($product['sale_price']); ?>
                                            </span>
                                            <span class="text-sm text-gray-500 line-through ml-2">
                                                <?php echo formatCurrency($product['price']); ?>
                                            </span>
                                        <?php else: ?>
                                            <span class="text-lg font-bold text-gray-900">
                                                <?php echo formatCurrency($product['price']); ?>
                                            </span>
                                        <?php endif; ?>
                                    </div>
                                    <button type="button"
                                            class="btn-primary w-full text-sm js-add-to-cart"
                                            data-product-id="<?php echo $product['id']; ?>">
                                        <i class="fas fa-shopping-bag mr-2"></i>Add to Cart
                                    </button>
                                </div>
                            </div>
                        <?php endwhile; ?>
                    </div>

                    <!-- Pagination -->
                    <?php if ($totalPages > 1): ?>
                        <div class="flex justify-center items-center space-x-2 mt-12">
                            <?php if ($page > 1): ?>
                                <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page - 1])); ?>" 
                                   class="btn-outline">
                                    <i class="fas fa-chevron-left"></i>
                                </a>
                            <?php endif; ?>

                            <?php for ($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
                                <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $i])); ?>" 
                                   class="px-4 py-2 rounded-lg <?php echo $i == $page ? 'bg-primary-600 text-white' : 'bg-white text-gray-700 hover:bg-gray-100'; ?> transition">
                                    <?php echo $i; ?>
                                </a>
                            <?php endfor; ?>

                            <?php if ($page < $totalPages): ?>
                                <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page + 1])); ?>" 
                                   class="btn-outline">
                                    <i class="fas fa-chevron-right"></i>
                                </a>
                            <?php endif; ?>
                        </div>
                    <?php endif; ?>
                <?php else: ?>
                    <div class="text-center py-20">
                        <div class="text-gray-400 text-6xl mb-4">
                            <i class="fas fa-search"></i>
                        </div>
                        <h3 class="text-2xl font-display font-bold text-gray-900 mb-2">No Products Found</h3>
                        <p class="text-gray-600 mb-6">Try adjusting your filters or search terms</p>
                        <a href="<?php echo SITE_URL; ?>/products.php" class="btn-primary">
                            View All Products
                        </a>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</section>

<?php 
closeDBConnection($conn);
require_once 'includes/footer.php'; 
?>
