<?php
require_once 'config/config.php';

$slug = isset($_GET['slug']) ? sanitize($_GET['slug']) : '';

if (!$slug) {
    redirect(SITE_URL . '/products.php');
}

$conn = getDBConnection();

// Get product details
$stmt = $conn->prepare("
    SELECT p.*, c.name as category_name, c.slug as category_slug
    FROM products p
    LEFT JOIN categories c ON p.category_id = c.id
    WHERE p.slug = ? AND p.status = 'active'
");
$stmt->bind_param("s", $slug);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows == 0) {
    $stmt->close();
    closeDBConnection($conn);
    redirect(SITE_URL . '/products.php');
}

$product = $result->fetch_assoc();
$stmt->close();

// Get product images
$imgStmt = $conn->prepare("
    SELECT * FROM product_images 
    WHERE product_id = ? 
    ORDER BY is_primary DESC, display_order ASC
");
$imgStmt->bind_param("i", $product['id']);
$imgStmt->execute();
$imagesResult = $imgStmt->get_result();
$productImages = [];
while ($img = $imagesResult->fetch_assoc()) {
    $productImages[] = $img;
}
$imgStmt->close();

$primaryImageUrl = getProductImage($productImages[0]['image_url'] ?? null, $product['id']);
$primaryImageIsPlaceholder = isPlaceholderImage($primaryImageUrl);

if (empty($productImages)) {
    $productImages[] = ['image_url' => null, 'placeholder' => getProductPlaceholder($product['id'])];
}

// Get related products
$relatedStmt = $conn->prepare("
    SELECT p.*, pi.image_url 
    FROM products p
    LEFT JOIN product_images pi ON p.id = pi.product_id AND pi.is_primary = 1
    WHERE p.category_id = ? AND p.id != ? AND p.status = 'active'
    ORDER BY RAND()
    LIMIT 4
");
$relatedStmt->bind_param("ii", $product['category_id'], $product['id']);
$relatedStmt->execute();
$relatedProducts = $relatedStmt->get_result();
$relatedStmt->close();

$pageTitle = htmlspecialchars($product['name']) . " | " . SITE_NAME;
require_once 'includes/header.php';
?>

<!-- Breadcrumb -->
<div class="bg-gray-50 py-4">
    <div class="container mx-auto px-4">
        <div class="flex items-center space-x-2 text-sm text-gray-600">
            <a href="<?php echo SITE_URL; ?>/index.php" class="hover:text-primary-600">Home</a>
            <i class="fas fa-chevron-right text-xs"></i>
            <a href="<?php echo SITE_URL; ?>/products.php" class="hover:text-primary-600">Products</a>
            <?php if ($product['category_name']): ?>
                <i class="fas fa-chevron-right text-xs"></i>
                <a href="<?php echo SITE_URL; ?>/products.php?category=<?php echo $product['category_slug']; ?>" 
                   class="hover:text-primary-600">
                    <?php echo htmlspecialchars($product['category_name']); ?>
                </a>
            <?php endif; ?>
            <i class="fas fa-chevron-right text-xs"></i>
            <span class="text-gray-900"><?php echo htmlspecialchars($product['name']); ?></span>
        </div>
    </div>
</div>

<!-- Product Details -->
<section class="py-12">
    <div class="container mx-auto px-4">
        <div class="grid md:grid-cols-2 gap-12">
            <!-- Product Images -->
            <div>
                <div class="sticky top-24">
                    <div id="mainImage" class="relative bg-gray-100 rounded-2xl overflow-hidden mb-4 aspect-square">
                        <img src="<?php echo $primaryImageUrl; ?>" 
                             alt="<?php echo htmlspecialchars($product['name']); ?>"
                             class="w-full h-full object-cover">
                        <?php if ($primaryImageIsPlaceholder): ?>
                            <div class="absolute inset-0 bg-gray-900/40 flex items-center justify-center">
                                <span class="text-white text-xs font-semibold tracking-wide uppercase">Image coming soon</span>
                            </div>
                        <?php endif; ?>
                    </div>
                    
                    <?php if (count($productImages) > 1): ?>
                        <div class="grid grid-cols-4 gap-4">
                            <?php foreach ($productImages as $img): 
                                $thumbUrl = getProductImage($img['image_url'] ?? null, $product['id'] . ($img['id'] ?? 'thumb'));
                                $thumbIsPlaceholder = isPlaceholderImage($thumbUrl);
                            ?>
                                <button onclick="changeImage('<?php echo $thumbUrl; ?>')" 
                                        class="relative bg-gray-100 rounded-lg overflow-hidden aspect-square hover:ring-2 hover:ring-primary-600 transition">
                                    <img src="<?php echo $thumbUrl; ?>" 
                                         alt="Product thumbnail"
                                         class="w-full h-full object-cover">
                                    <?php if ($thumbIsPlaceholder): ?>
                                        <span class="absolute inset-x-2 bottom-2 text-[10px] font-semibold text-white uppercase tracking-wide bg-gray-900/70 px-2 py-1 rounded">Coming soon</span>
                                    <?php endif; ?>
                                </button>
                            <?php endforeach; ?>
                        </div>
                    <?php else: ?>
                        <div class="grid grid-cols-4 gap-4">
                            <?php for ($i = 1; $i <= 4; $i++): 
                                $placeholderThumb = getProductPlaceholder($product['id'] . '-thumb-' . $i);
                            ?>
                                <button onclick="changeImage('<?php echo $placeholderThumb; ?>')" 
                                        class="relative bg-gray-100 rounded-lg overflow-hidden aspect-square hover:ring-2 hover:ring-primary-600 transition">
                                    <img src="<?php echo $placeholderThumb; ?>" 
                                         alt="Product thumbnail"
                                         class="w-full h-full object-cover">
                                    <span class="absolute inset-x-2 bottom-2 text-[10px] font-semibold text-white uppercase tracking-wide bg-gray-900/70 px-2 py-1 rounded">Coming soon</span>
                                </button>
                            <?php endfor; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Product Info -->
            <div>
                <?php if ($product['category_name']): ?>
                    <a href="<?php echo SITE_URL; ?>/products.php?category=<?php echo $product['category_slug']; ?>" 
                       class="inline-block text-sm text-primary-600 hover:text-primary-700 font-medium mb-2">
                        <?php echo htmlspecialchars($product['category_name']); ?>
                    </a>
                <?php endif; ?>
                
                <h1 class="text-4xl md:text-5xl font-display font-bold text-gray-900 mb-4">
                    <?php echo htmlspecialchars($product['name']); ?>
                </h1>

                <div class="flex items-center space-x-4 mb-6">
                    <?php if ($product['sale_price']): ?>
                        <span class="text-4xl font-bold text-primary-600">
                            <?php echo formatCurrency($product['sale_price']); ?>
                        </span>
                        <span class="text-2xl text-gray-500 line-through">
                            <?php echo formatCurrency($product['price']); ?>
                        </span>
                        <span class="badge badge-sale text-sm">
                            Save <?php echo round((($product['price'] - $product['sale_price']) / $product['price']) * 100); ?>%
                        </span>
                    <?php else: ?>
                        <span class="text-4xl font-bold text-gray-900">
                            <?php echo formatCurrency($product['price']); ?>
                        </span>
                    <?php endif; ?>
                </div>

                <div class="flex flex-wrap gap-2 mb-6">
                    <?php if ($product['new_arrival']): ?>
                        <span class="badge badge-new">New Arrival</span>
                    <?php endif; ?>
                    <?php if ($product['featured']): ?>
                        <span class="badge badge-featured">Featured</span>
                    <?php endif; ?>
                    <?php if ($product['stock_quantity'] > 0): ?>
                        <span class="badge bg-green-100 text-green-800">
                            <i class="fas fa-check-circle mr-1"></i> In Stock
                        </span>
                    <?php else: ?>
                        <span class="badge bg-red-100 text-red-800">
                            <i class="fas fa-times-circle mr-1"></i> Out of Stock
                        </span>
                    <?php endif; ?>
                </div>

                <?php if ($product['short_description']): ?>
                    <p class="text-lg text-gray-600 mb-6 leading-relaxed">
                        <?php echo nl2br(htmlspecialchars($product['short_description'])); ?>
                    </p>
                <?php endif; ?>

                <!-- Quantity & Add to Cart -->
                <div class="bg-gray-50 rounded-xl p-6 mb-6">
                    <div class="flex items-center space-x-4 mb-4">
                        <label class="font-semibold text-gray-900">Quantity:</label>
                        <div class="flex items-center border border-gray-300 rounded-lg">
                            <button onclick="decreaseQty()" class="px-4 py-2 hover:bg-gray-100 transition">
                                <i class="fas fa-minus"></i>
                            </button>
                            <input type="number" id="quantity" value="1" min="1" max="<?php echo $product['stock_quantity']; ?>" 
                                   class="w-16 text-center border-x border-gray-300 py-2 focus:outline-none">
                            <button onclick="increaseQty()" class="px-4 py-2 hover:bg-gray-100 transition">
                                <i class="fas fa-plus"></i>
                            </button>
                        </div>
                    </div>

                    <?php if ($product['stock_quantity'] > 0): ?>
                        <button type="button"
                                class="btn-primary w-full text-lg mb-3 js-add-to-cart"
                                data-product-id="<?php echo $product['id']; ?>"
                                data-quantity-input="quantity">
                            <i class="fas fa-shopping-bag mr-2"></i>Add to Cart
                        </button>
                    <?php else: ?>
                        <button disabled class="btn-primary w-full text-lg mb-3 opacity-50 cursor-not-allowed">
                            <i class="fas fa-times-circle mr-2"></i>Out of Stock
                        </button>
                    <?php endif; ?>
                    
                    <button onclick="addToWishlist(<?php echo $product['id']; ?>)" 
                            class="btn-outline w-full text-lg">
                        <i class="far fa-heart mr-2"></i>Add to Wishlist
                    </button>
                </div>

                <!-- Product Features -->
                <div class="grid grid-cols-2 gap-4 mb-6">
                    <div class="flex items-start space-x-3">
                        <i class="fas fa-leaf text-primary-600 text-xl mt-1"></i>
                        <div>
                            <p class="font-semibold text-gray-900">100% Organic</p>
                            <p class="text-sm text-gray-600">Natural ingredients</p>
                        </div>
                    </div>
                    <div class="flex items-start space-x-3">
                        <i class="fas fa-heart text-primary-600 text-xl mt-1"></i>
                        <div>
                            <p class="font-semibold text-gray-900">Cruelty-Free</p>
                            <p class="text-sm text-gray-600">Never tested on animals</p>
                        </div>
                    </div>
                    <div class="flex items-start space-x-3">
                        <i class="fas fa-shipping-fast text-primary-600 text-xl mt-1"></i>
                        <div>
                            <p class="font-semibold text-gray-900">Fast Delivery</p>
                            <p class="text-sm text-gray-600">3-5 business days</p>
                        </div>
                    </div>
                    <div class="flex items-start space-x-3">
                        <i class="fas fa-undo text-primary-600 text-xl mt-1"></i>
                        <div>
                            <p class="font-semibold text-gray-900">Easy Returns</p>
                            <p class="text-sm text-gray-600">30-day guarantee</p>
                        </div>
                    </div>
                </div>

                <!-- Product Description -->
                <?php if ($product['description']): ?>
                    <div class="border-t border-gray-200 pt-6">
                        <h2 class="text-2xl font-display font-bold text-gray-900 mb-4">Product Description</h2>
                        <div class="prose max-w-none text-gray-600 leading-relaxed">
                            <?php echo nl2br(htmlspecialchars($product['description'])); ?>
                        </div>
                    </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- Related Products -->
        <?php if ($relatedProducts->num_rows > 0): ?>
            <div class="mt-20">
                <h2 class="text-3xl font-display font-bold text-gray-900 mb-8">You May Also Like</h2>
                <div class="grid grid-cols-2 md:grid-cols-4 gap-6">
                    <?php while ($relatedProduct = $relatedProducts->fetch_assoc()): ?>
                        <div class="card product-card group">
                            <a href="<?php echo SITE_URL; ?>/product.php?slug=<?php echo $relatedProduct['slug']; ?>" class="block">
                                <div class="relative overflow-hidden bg-gray-100 aspect-square">
                                    <?php 
                                        $relatedImage = getProductImage($relatedProduct['image_url'], $relatedProduct['id']);
                                        $relatedImageIsPlaceholder = isPlaceholderImage($relatedImage);
                                    ?>
                                    <img src="<?php echo $relatedImage; ?>" 
                                         alt="<?php echo htmlspecialchars($relatedProduct['name']); ?>"
                                         class="w-full h-full object-cover">
                                    <?php if ($relatedImageIsPlaceholder): ?>
                                        <div class="absolute inset-0 bg-gray-900/40 flex items-center justify-center">
                                            <span class="text-white text-xs font-semibold tracking-wide uppercase">Image coming soon</span>
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </a>
                            <div class="p-4">
                                <a href="<?php echo SITE_URL; ?>/product.php?slug=<?php echo $relatedProduct['slug']; ?>">
                                    <h3 class="font-semibold text-gray-900 mb-2 line-clamp-2 group-hover:text-primary-600 transition">
                                        <?php echo htmlspecialchars($relatedProduct['name']); ?>
                                    </h3>
                                </a>
                                <div class="text-lg font-bold text-gray-900">
                                    <?php echo formatCurrency($relatedProduct['sale_price'] ?? $relatedProduct['price']); ?>
                                </div>
                            </div>
                        </div>
                    <?php endwhile; ?>
                </div>
            </div>
        <?php endif; ?>
    </div>
</section>

<script>
function changeImage(src) {
    document.querySelector('#mainImage img').src = src;
}

function increaseQty() {
    const qtyInput = document.getElementById('quantity');
    const max = parseInt(qtyInput.max);
    const current = parseInt(qtyInput.value);
    if (current < max) {
        qtyInput.value = current + 1;
    }
}

function decreaseQty() {
    const qtyInput = document.getElementById('quantity');
    const current = parseInt(qtyInput.value);
    if (current > 1) {
        qtyInput.value = current - 1;
    }
}

function addToWishlist(productId) {
    alert('Wishlist feature coming soon!');
}
</script>

<?php 
closeDBConnection($conn);
require_once 'includes/footer.php'; 
?>
