-- Skincare E-commerce Database Schema
CREATE DATABASE IF NOT EXISTS skincare_shop CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE skincare_shop;

-- Users table
CREATE TABLE IF NOT EXISTS users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    email VARCHAR(100) UNIQUE NOT NULL,
    password VARCHAR(255) NOT NULL,
    phone VARCHAR(20),
    address TEXT,
    city VARCHAR(50),
    postal_code VARCHAR(10),
    role ENUM('customer', 'admin') DEFAULT 'customer',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Categories table
CREATE TABLE IF NOT EXISTS categories (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    slug VARCHAR(100) UNIQUE NOT NULL,
    description TEXT,
    image VARCHAR(255),
    display_order INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Products table
CREATE TABLE IF NOT EXISTS products (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(200) NOT NULL,
    slug VARCHAR(200) UNIQUE NOT NULL,
    description TEXT,
    short_description VARCHAR(500),
    price DECIMAL(10, 2) NOT NULL,
    sale_price DECIMAL(10, 2),
    category_id INT,
    stock_quantity INT DEFAULT 0,
    sku VARCHAR(50) UNIQUE,
    featured BOOLEAN DEFAULT FALSE,
    new_arrival BOOLEAN DEFAULT FALSE,
    status ENUM('active', 'inactive') DEFAULT 'active',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (category_id) REFERENCES categories(id) ON DELETE SET NULL
);

-- Product images table
CREATE TABLE IF NOT EXISTS product_images (
    id INT AUTO_INCREMENT PRIMARY KEY,
    product_id INT NOT NULL,
    image_url VARCHAR(255) NOT NULL,
    is_primary BOOLEAN DEFAULT FALSE,
    display_order INT DEFAULT 0,
    FOREIGN KEY (product_id) REFERENCES products(id) ON DELETE CASCADE
);

-- Orders table
CREATE TABLE IF NOT EXISTS orders (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT,
    order_number VARCHAR(50) UNIQUE NOT NULL,
    total_amount DECIMAL(10, 2) NOT NULL,
    status ENUM('pending', 'processing', 'shipped', 'delivered', 'cancelled') DEFAULT 'pending',
    payment_method VARCHAR(50),
    payment_status ENUM('pending', 'paid', 'failed') DEFAULT 'pending',
    shipping_name VARCHAR(100) NOT NULL,
    shipping_email VARCHAR(100) NOT NULL,
    shipping_phone VARCHAR(20),
    shipping_address TEXT NOT NULL,
    shipping_city VARCHAR(50) NOT NULL,
    shipping_postal_code VARCHAR(10),
    notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE SET NULL
);

-- Order items table
CREATE TABLE IF NOT EXISTS order_items (
    id INT AUTO_INCREMENT PRIMARY KEY,
    order_id INT NOT NULL,
    product_id INT,
    product_name VARCHAR(200) NOT NULL,
    product_price DECIMAL(10, 2) NOT NULL,
    quantity INT NOT NULL,
    subtotal DECIMAL(10, 2) NOT NULL,
    FOREIGN KEY (order_id) REFERENCES orders(id) ON DELETE CASCADE,
    FOREIGN KEY (product_id) REFERENCES products(id) ON DELETE SET NULL
);

-- Cart table
CREATE TABLE IF NOT EXISTS cart (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT,
    session_id VARCHAR(255),
    product_id INT NOT NULL,
    quantity INT NOT NULL DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (product_id) REFERENCES products(id) ON DELETE CASCADE
);

-- Hero banner settings table
CREATE TABLE IF NOT EXISTS hero_banners (
    id INT AUTO_INCREMENT PRIMARY KEY,
    title VARCHAR(200) NOT NULL,
    subtitle VARCHAR(500),
    button_text VARCHAR(100),
    button_link TEXT,
    image_url TEXT,
    secondary_image_url TEXT,
    bg_position_desktop VARCHAR(50) DEFAULT 'center',
    bg_position_mobile VARCHAR(50) DEFAULT 'center',
    is_active BOOLEAN DEFAULT TRUE,
    display_order INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Reviews table
CREATE TABLE IF NOT EXISTS reviews (
    id INT AUTO_INCREMENT PRIMARY KEY,
    product_id INT NOT NULL,
    user_id INT,
    rating INT NOT NULL CHECK (rating >= 1 AND rating <= 5),
    review_text TEXT,
    status ENUM('pending', 'approved', 'rejected') DEFAULT 'pending',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (product_id) REFERENCES products(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE SET NULL
);

-- Insert default admin users
-- Admin User: admin@skincare.com / admin123
-- Admin: admin@admin.com / Admin
INSERT INTO users (name, email, password, role) VALUES 
('Admin User', 'admin@skincare.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'admin'),
('Admin', 'admin@admin.com', '$2y$10$/lHoGDZdSA7PSpc9CB7uIuXS5keuFmLji626VfYdwaGTpsYxjSD02', 'admin');

-- Insert sample categories
INSERT INTO categories (name, slug, description, image, display_order) VALUES
('Cleansers', 'cleansers', 'Gentle cleansers for all skin types', 'cleansers.jpg', 1),
('Serums', 'serums', 'Powerful serums for targeted treatments', 'serums.jpg', 2),
('Moisturizers', 'moisturizers', 'Hydrating creams and lotions', 'moisturizers.jpg', 3),
('Masks', 'masks', 'Deep treatment masks', 'masks.jpg', 4),
('Eye Care', 'eye-care', 'Specialized eye treatments', 'eye-care.jpg', 5),
('Sun Protection', 'sun-protection', 'SPF and sun care products', 'sun-protection.jpg', 6);

-- Insert sample products
INSERT INTO products (name, slug, description, short_description, price, sale_price, category_id, stock_quantity, sku, featured, new_arrival, status) VALUES
('Apricot Calendula Nourishing Cream', 'apricot-calendula-nourishing-cream', 'A luxurious moisturizer enriched with apricot and calendula to deeply nourish and hydrate sensitive skin. Perfect for daily use.', 'Nourishing cream for sensitive skin', 1299.00, NULL, 3, 50, 'ACCN001', TRUE, TRUE, 'active'),
('Lemon Grass Cleanser', 'lemon-grass-cleanser', 'Refreshing gel cleanser with lemongrass extract. Deeply cleanses pores while maintaining skin\'s natural moisture balance.', 'Refreshing gel cleanser', 899.00, 799.00, 1, 75, 'LGC001', TRUE, FALSE, 'active'),
('Cornflower Recovery Serum', 'cornflower-recovery-serum', 'Advanced recovery serum with cornflower extract. Reduces redness and strengthens sensitive skin barrier.', 'Recovery serum for sensitive skin', 1599.00, NULL, 2, 30, 'CRS001', TRUE, TRUE, 'active'),
('Charcoal & Black Seed Clay Masque', 'charcoal-black-seed-clay-masque', 'Deep cleansing mask with activated charcoal and black seed oil. Detoxifies and purifies congested pores.', 'Deep cleansing clay mask', 1099.00, NULL, 4, 45, 'CBSCM001', FALSE, TRUE, 'active'),
('Bearberry Eye Repair Cream', 'bearberry-eye-repair-cream', 'Intensive eye cream with bearberry extract. Reduces dark circles, puffiness, and fine lines around delicate eye area.', 'Intensive eye repair cream', 1399.00, 1199.00, 5, 40, 'BERC001', TRUE, FALSE, 'active'),
('Daily Defense Tinted SPF', 'daily-defense-tinted-spf', 'Lightweight tinted sunscreen with SPF 30. Provides protection while evening out skin tone naturally.', 'Tinted SPF 30 sunscreen', 999.00, NULL, 6, 60, 'DDTS001', FALSE, TRUE, 'active'),
('Facial Recovery Oil', 'facial-recovery-oil', 'Nourishing facial oil blend with precious botanicals. Restores radiance and strengthens skin barrier overnight.', 'Nourishing facial oil', 1499.00, NULL, 2, 35, 'FRO001', TRUE, FALSE, 'active'),
('Mango Exfoliating Enzyme Masque', 'mango-exfoliating-enzyme-masque', 'Gentle enzymatic mask with mango extract. Exfoliates dead skin cells revealing brighter, smoother complexion.', 'Gentle exfoliating mask', 1199.00, 999.00, 4, 50, 'MEEM001', FALSE, TRUE, 'active'),
('Echinacea Recovery Cream', 'echinacea-recovery-cream', 'Soothing cream with echinacea extract. Calms irritated skin and supports natural healing process.', 'Soothing recovery cream', 1349.00, NULL, 3, 42, 'ERC001', FALSE, FALSE, 'active'),
('Hawthorn Tonique', 'hawthorn-tonique', 'Balancing toner with hawthorn extract. Refines pores and prepares skin for better product absorption.', 'Balancing toner', 849.00, NULL, 1, 65, 'HT001', FALSE, FALSE, 'active'),
('Barbados Cherry Enzyme Cleansing Powder', 'barbados-cherry-enzyme-cleansing-powder', 'Unique powder cleanser with Barbados cherry. Transforms into creamy foam for gentle yet effective cleansing.', 'Enzyme cleansing powder', 949.00, NULL, 1, 55, 'BCECP001', FALSE, TRUE, 'active'),
('Charcoal & Black Seed Clarifying Oil', 'charcoal-black-seed-clarifying-oil', 'Lightweight clarifying oil with charcoal and black seed. Balances oil production while deeply nourishing skin.', 'Clarifying facial oil', 1299.00, NULL, 2, 38, 'CBSCO001', FALSE, FALSE, 'active');

-- Insert sample product images
INSERT INTO product_images (product_id, image_url, is_primary, display_order) VALUES
(1, 'apricot-calendula-cream.jpg', TRUE, 1),
(2, 'lemon-grass-cleanser.jpg', TRUE, 1),
(3, 'cornflower-serum.jpg', TRUE, 1),
(4, 'charcoal-mask.jpg', TRUE, 1),
(5, 'bearberry-eye-cream.jpg', TRUE, 1),
(6, 'daily-defense-spf.jpg', TRUE, 1),
(7, 'facial-recovery-oil.jpg', TRUE, 1),
(8, 'mango-enzyme-mask.jpg', TRUE, 1),
(9, 'echinacea-cream.jpg', TRUE, 1),
(10, 'hawthorn-tonique.jpg', TRUE, 1),
(11, 'barbados-cherry-powder.jpg', TRUE, 1),
(12, 'charcoal-clarifying-oil.jpg', TRUE, 1);

-- Insert default hero banner
INSERT INTO hero_banners (title, subtitle, button_text, button_link, image_url, secondary_image_url, is_active, display_order) VALUES
('Discover Natural Beauty', 'Premium organic skincare products crafted with the finest natural ingredients for radiant, healthy skin', 'Shop Now', '/skin/products.php', 'https://images.unsplash.com/photo-1556228720-195a672e8a03?w=1920&h=800&fit=crop', 'https://images.unsplash.com/photo-1522335789205-197f1c2b6a86?auto=format&fit=crop&w=800&q=80', TRUE, 1);
