<?php
// Start session if not already started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Site configuration
define('SITE_NAME', 'Skin Care');

// Auto-detect site URL based on server environment
$protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') || 
            (!empty($_SERVER['SERVER_PORT']) && $_SERVER['SERVER_PORT'] == 443) ? 'https://' : 'http://';
$host = $_SERVER['HTTP_HOST'] ?? 'localhost';
$scriptPath = rtrim(dirname($_SERVER['SCRIPT_NAME']), '/');
// Remove common subdirectories from the path (config, includes, api, admin)
$basePath = preg_replace('/(\/config|\/includes|\/api|\/admin)$/', '', $scriptPath);
define('SITE_URL', $protocol . $host . $basePath);

define('CURRENCY', 'R');
define('FREE_SHIPPING_THRESHOLD', 2000);

// File upload configuration
define('UPLOAD_DIR', __DIR__ . '/../uploads/');
define('MAX_FILE_SIZE', 5242880); // 5MB

// Pagination
define('PRODUCTS_PER_PAGE', 12);

// Include database configuration
require_once __DIR__ . '/database.php';

// Helper function to check if user is logged in
function isLoggedIn() {
    return isset($_SESSION['user_id']);
}

// Helper function to check if user is admin
function isAdmin() {
    return isset($_SESSION['user_role']) && $_SESSION['user_role'] === 'admin';
}

// Helper function to get current user ID
function getCurrentUserId() {
    return $_SESSION['user_id'] ?? null;
}

// Helper function to redirect
function redirect($url) {
    header("Location: $url");
    exit();
}

// Helper function to format currency
function formatCurrency($amount) {
    return CURRENCY . number_format($amount, 2);
}

// Helper function to sanitize input
function sanitize($data) {
    return htmlspecialchars(strip_tags(trim($data)));
}

// Helper function to get the base path for assets (works from any subdirectory)
function getBasePath() {
    $scriptDir = dirname($_SERVER['SCRIPT_NAME']);
    // Count how deep we are (e.g., /skin/admin = 1 level deep, /skin = 0 levels)
    $currentPath = trim($scriptDir, '/');
    $parts = explode('/', $currentPath);
    
    // If we're in a subdirectory (admin, api, includes), go up one level
    $knownSubdirs = ['admin', 'api', 'includes'];
    if (count($parts) > 1 && in_array(end($parts), $knownSubdirs)) {
        array_pop($parts);
    }
    
    return '/' . implode('/', $parts);
}

// Helper function to generate asset URL
function assetUrl($path) {
    $basePath = getBasePath();
    $path = ltrim($path, '/');
    return $basePath . '/' . $path;
}

// Helper function to get cart session ID
function getCartSessionId() {
    if (!isset($_SESSION['cart_session_id'])) {
        $_SESSION['cart_session_id'] = bin2hex(random_bytes(16));
    }
    return $_SESSION['cart_session_id'];
}

// Helper function to generate order number
function generateOrderNumber() {
    return 'ORD-' . date('Ymd') . '-' . strtoupper(substr(bin2hex(random_bytes(4)), 0, 8));
}

function getProductPlaceholder($seed = null) {
    static $images = [
        'https://images.unsplash.com/photo-1522335789203-aabd1fc54bc9?auto=format&fit=crop&w=800&q=80',
        'https://images.unsplash.com/photo-1522336572468-97b06e8ef143?auto=format&fit=crop&w=800&q=80',
        'https://images.unsplash.com/photo-1582719478250-59322126687b?auto=format&fit=crop&w=800&q=80',
        'https://images.unsplash.com/photo-1612810806695-30b44b1f1c6d?auto=format&fit=crop&w=800&q=80',
        'https://images.unsplash.com/photo-1522337956268-607c5f01e0e3?auto=format&fit=crop&w=800&q=80',
        'https://images.unsplash.com/photo-1540555700478-4be289fbecef?auto=format&fit=crop&w=800&q=80',
        'https://images.unsplash.com/photo-1527515637462-6dbad0f4df32?auto=format&fit=crop&w=800&q=80',
        'https://images.unsplash.com/photo-1522335789205-197f1c2b6a86?auto=format&fit=crop&w=800&q=80',
    ];

    if ($seed === null) {
        return $images[array_rand($images)];
    }

    $index = abs(crc32((string) $seed)) % count($images);
    return $images[$index];
}

function getCategoryPlaceholder($slug = '') {
    $placeholders = [
        'cleansers' => 'https://images.unsplash.com/photo-1522336572468-97b06e8ef143?auto=format&fit=crop&w=800&q=80',
        'serums' => 'https://images.unsplash.com/photo-1524504388940-b1c1722653e1?auto=format&fit=crop&w=800&q=80',
        'moisturizers' => 'https://images.unsplash.com/photo-1522337660859-02fbefca4702?auto=format&fit=crop&w=800&q=80',
        'masks' => 'https://images.unsplash.com/photo-1540555700478-4be289fbecef?auto=format&fit=crop&w=800&q=80',
        'eye-care' => 'https://images.unsplash.com/photo-1527515637462-6dbad0f4df32?auto=format&fit=crop&w=800&q=80',
        'sun-protection' => 'https://images.unsplash.com/photo-1522335789203-aabd1fc54bc9?auto=format&fit=crop&w=800&q=80',
        'default' => 'https://images.unsplash.com/photo-1522337956268-607c5f01e0e3?auto=format&fit=crop&w=800&q=80',
    ];

    $slugKey = strtolower((string) $slug);
    return $placeholders[$slugKey] ?? $placeholders['default'];
}

function resolveUploadedImageUrl($imageFilename, $subDirectory) {
    if (empty($imageFilename)) {
        return null;
    }

    if (filter_var($imageFilename, FILTER_VALIDATE_URL)) {
        return $imageFilename;
    }

    $sanitized = str_replace(['..', '\\'], '', trim((string) $imageFilename));
    if ($sanitized === '') {
        return null;
    }

    $sanitized = ltrim($sanitized, '/');
    $sanitized = strtok($sanitized, '?');

    $knownPrefixes = [
        'uploads/' . $subDirectory . '/',
        $subDirectory . '/',
    ];

    foreach ($knownPrefixes as $prefix) {
        if (strpos($sanitized, $prefix) === 0) {
            $sanitized = substr($sanitized, strlen($prefix));
            break;
        }
    }

    $uploadBase = rtrim(UPLOAD_DIR, '/\\') . DIRECTORY_SEPARATOR;
    $normalizedRelative = str_replace('\\', '/', $sanitized);
    $normalizedRelative = ltrim($normalizedRelative, '/');

    $pathsToCheck = [];
    $pathsToCheck[] = [
        $uploadBase . $subDirectory . DIRECTORY_SEPARATOR . str_replace('/', DIRECTORY_SEPARATOR, $normalizedRelative),
        SITE_URL . '/uploads/' . $subDirectory . '/' . $normalizedRelative,
    ];
    $pathsToCheck[] = [
        $uploadBase . str_replace('/', DIRECTORY_SEPARATOR, $sanitized),
        SITE_URL . '/uploads/' . $sanitized,
    ];

    foreach ($pathsToCheck as [$filePath, $publicUrl]) {
        if ($normalizedRelative && file_exists($filePath)) {
            return $publicUrl;
        }
    }

    return null;
}

function getProductImage($imageFilename = null, $seed = null) {
    $resolvedUrl = resolveUploadedImageUrl($imageFilename, 'products');

    if ($resolvedUrl) {
        return $resolvedUrl;
    }

    return getProductPlaceholder($seed);
}

function getCategoryImage($imageFilename = null, $slug = '') {
    $resolvedUrl = resolveUploadedImageUrl($imageFilename, 'categories');

    if ($resolvedUrl) {
        return $resolvedUrl;
    }

    return getCategoryPlaceholder($slug);
}

function isPlaceholderImage($imageUrl) {
    if (empty($imageUrl)) {
        return false;
    }

    return (bool) preg_match('/images\.unsplash\.com/i', $imageUrl);
}

function slugify($text) {
    $text = preg_replace('~[\pL\d]+~u', ' $0 ', $text);
    $text = iconv('utf-8', 'us-ascii//TRANSLIT', $text);
    $text = preg_replace('~[^\w]+~', '-', strtolower($text));
    $text = trim($text, '-');
    return $text ?: uniqid('item-');
}

function saveUploadedImage(array $file, $subDirectory = 'products') {
    if (empty($file['name']) || $file['error'] !== UPLOAD_ERR_OK) {
        return null;
    }

    $allowedMime = ['image/jpeg' => 'jpg', 'image/png' => 'png', 'image/webp' => 'webp'];
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    $mimeType = finfo_file($finfo, $file['tmp_name']);
    finfo_close($finfo);

    if (!isset($allowedMime[$mimeType])) {
        return null;
    }

    $extension = $allowedMime[$mimeType];
    $fileName = slugify(pathinfo($file['name'], PATHINFO_FILENAME)) . '-' . time() . '.' . $extension;
    $targetDir = UPLOAD_DIR . $subDirectory . '/';

    if (!is_dir($targetDir)) {
        mkdir($targetDir, 0755, true);
    }

    $targetPath = $targetDir . $fileName;
    if (!move_uploaded_file($file['tmp_name'], $targetPath)) {
        return null;
    }

    return $fileName;
}
