<?php
/**
 * Setup Verification Script
 * Run this file to check if everything is configured correctly
 */

$checks = [];
$errors = [];
$warnings = [];

// Check PHP version
$phpVersion = phpversion();
$checks[] = [
    'name' => 'PHP Version',
    'status' => version_compare($phpVersion, '7.4.0', '>='),
    'message' => "PHP $phpVersion " . (version_compare($phpVersion, '7.4.0', '>=') ? '✓' : '✗ (Need 7.4+)'),
];

// Check if database config exists
$checks[] = [
    'name' => 'Database Config',
    'status' => file_exists(__DIR__ . '/config/database.php'),
    'message' => file_exists(__DIR__ . '/config/database.php') ? '✓ Found' : '✗ Missing',
];

// Check if CSS is compiled
$cssExists = file_exists(__DIR__ . '/assets/css/output.css');
$checks[] = [
    'name' => 'Compiled CSS',
    'status' => $cssExists,
    'message' => $cssExists ? '✓ Found' : '✗ Run: npm run build:css',
];

// Check uploads directory
$uploadsWritable = is_writable(__DIR__ . '/uploads/products');
$checks[] = [
    'name' => 'Uploads Directory',
    'status' => $uploadsWritable,
    'message' => $uploadsWritable ? '✓ Writable' : '✗ Not writable (chmod 755)',
];

// Check database connection
try {
    require_once __DIR__ . '/config/database.php';
    $conn = @getDBConnection();
    if ($conn) {
        $checks[] = [
            'name' => 'Database Connection',
            'status' => true,
            'message' => '✓ Connected',
        ];
        
        // Check if tables exist
        $tables = ['users', 'products', 'categories', 'cart', 'orders'];
        $existingTables = [];
        $result = $conn->query("SHOW TABLES");
        while ($row = $result->fetch_array()) {
            $existingTables[] = $row[0];
        }
        
        $missingTables = array_diff($tables, $existingTables);
        $checks[] = [
            'name' => 'Database Tables',
            'status' => empty($missingTables),
            'message' => empty($missingTables) ? '✓ All tables exist' : '✗ Missing: ' . implode(', ', $missingTables),
        ];
        
        // Check if sample data exists
        $productCount = $conn->query("SELECT COUNT(*) as count FROM products")->fetch_assoc()['count'];
        $checks[] = [
            'name' => 'Sample Products',
            'status' => $productCount > 0,
            'message' => $productCount > 0 ? "✓ $productCount products loaded" : '⚠ No products (import database.sql)',
        ];
        
        closeDBConnection($conn);
    } else {
        $checks[] = [
            'name' => 'Database Connection',
            'status' => false,
            'message' => '✗ Failed to connect',
        ];
    }
} catch (Exception $e) {
    $checks[] = [
        'name' => 'Database Connection',
        'status' => false,
        'message' => '✗ Error: ' . $e->getMessage(),
    ];
}

// Check required PHP extensions
$extensions = ['mysqli', 'session', 'json'];
foreach ($extensions as $ext) {
    $loaded = extension_loaded($ext);
    $checks[] = [
        'name' => "PHP Extension: $ext",
        'status' => $loaded,
        'message' => $loaded ? '✓ Loaded' : "✗ Missing",
    ];
}

// Calculate success rate
$passed = count(array_filter($checks, fn($c) => $c['status']));
$total = count($checks);
$percentage = round(($passed / $total) * 100);

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Setup Verification - Organic Skincare Shop</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        .container {
            background: white;
            border-radius: 20px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3);
            max-width: 800px;
            width: 100%;
            overflow: hidden;
        }
        .header {
            background: linear-gradient(135deg, #16a34a 0%, #15803d 100%);
            color: white;
            padding: 40px;
            text-align: center;
        }
        .header h1 {
            font-size: 32px;
            margin-bottom: 10px;
        }
        .header p {
            opacity: 0.9;
            font-size: 18px;
        }
        .progress-bar {
            background: rgba(255,255,255,0.2);
            height: 8px;
            border-radius: 10px;
            margin-top: 20px;
            overflow: hidden;
        }
        .progress-fill {
            background: white;
            height: 100%;
            transition: width 1s ease;
            border-radius: 10px;
        }
        .content {
            padding: 40px;
        }
        .check-item {
            padding: 15px 20px;
            margin-bottom: 10px;
            border-radius: 10px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            transition: all 0.3s;
        }
        .check-item:hover {
            transform: translateX(5px);
        }
        .check-item.success {
            background: #f0fdf4;
            border-left: 4px solid #16a34a;
        }
        .check-item.error {
            background: #fef2f2;
            border-left: 4px solid #dc2626;
        }
        .check-name {
            font-weight: 600;
            color: #1f2937;
        }
        .check-message {
            font-family: 'Courier New', monospace;
            font-size: 14px;
        }
        .success .check-message { color: #16a34a; }
        .error .check-message { color: #dc2626; }
        .summary {
            background: #f9fafb;
            padding: 30px;
            border-radius: 10px;
            text-align: center;
            margin-top: 30px;
        }
        .summary h2 {
            color: #1f2937;
            margin-bottom: 10px;
        }
        .percentage {
            font-size: 48px;
            font-weight: bold;
            color: <?php echo $percentage >= 80 ? '#16a34a' : ($percentage >= 50 ? '#f59e0b' : '#dc2626'); ?>;
        }
        .actions {
            display: flex;
            gap: 10px;
            margin-top: 20px;
            justify-content: center;
        }
        .btn {
            padding: 12px 24px;
            border-radius: 8px;
            text-decoration: none;
            font-weight: 600;
            transition: all 0.3s;
            display: inline-block;
        }
        .btn-primary {
            background: #16a34a;
            color: white;
        }
        .btn-primary:hover {
            background: #15803d;
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(22, 163, 74, 0.4);
        }
        .btn-secondary {
            background: #e5e7eb;
            color: #1f2937;
        }
        .btn-secondary:hover {
            background: #d1d5db;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>🌿 Setup Verification</h1>
            <p>Organic Skincare Shop</p>
            <div class="progress-bar">
                <div class="progress-fill" style="width: <?php echo $percentage; ?>%"></div>
            </div>
        </div>
        
        <div class="content">
            <h2 style="margin-bottom: 20px; color: #1f2937;">System Checks</h2>
            
            <?php foreach ($checks as $check): ?>
                <div class="check-item <?php echo $check['status'] ? 'success' : 'error'; ?>">
                    <span class="check-name"><?php echo $check['name']; ?></span>
                    <span class="check-message"><?php echo $check['message']; ?></span>
                </div>
            <?php endforeach; ?>
            
            <div class="summary">
                <h2>Overall Status</h2>
                <div class="percentage"><?php echo $percentage; ?>%</div>
                <p style="color: #6b7280; margin-top: 10px;">
                    <?php 
                    if ($percentage == 100) {
                        echo "🎉 Perfect! Everything is set up correctly!";
                    } elseif ($percentage >= 80) {
                        echo "✅ Good! Minor issues detected.";
                    } elseif ($percentage >= 50) {
                        echo "⚠️ Some issues need attention.";
                    } else {
                        echo "❌ Critical issues detected. Please fix them.";
                    }
                    ?>
                </p>
                
                <div class="actions">
                    <?php if ($percentage >= 80): ?>
                        <a href="index.php" class="btn btn-primary">Go to Website</a>
                    <?php endif; ?>
                    <a href="README.md" class="btn btn-secondary">View Documentation</a>
                    <a href="?refresh=1" class="btn btn-secondary">Refresh Check</a>
                </div>
            </div>
        </div>
    </div>
</body>
</html>
