<?php
require_once __DIR__ . '/../config/config.php';

if (!isLoggedIn()) {
    $_SESSION['admin_redirect'] = true;
    redirect(SITE_URL . '/login.php?redirect=admin');
} elseif (!isAdmin()) {
    redirect(SITE_URL . '/index.php');
}

$activePage = 'orders';
$pageTitle = 'Order Management | ' . SITE_NAME;

$conn = getDBConnection();

$statusFilter = $_GET['status'] ?? '';
$allowedStatuses = ['pending', 'processing', 'shipped', 'delivered', 'cancelled'];

$query = "
    SELECT o.*, u.name AS customer_name
    FROM orders o
    LEFT JOIN users u ON o.user_id = u.id
";

$params = [];
$types = '';
if (in_array($statusFilter, $allowedStatuses, true)) {
    $query .= " WHERE o.status = ?";
    $params[] = $statusFilter;
    $types .= 's';
}

$query .= " ORDER BY o.created_at DESC";

$stmt = $conn->prepare($query);
if ($params) {
    $stmt->bind_param($types, ...$params);
}
$stmt->execute();
$orders = $stmt->get_result();

$orderDetails = null;
if (isset($_GET['order_id'])) {
    $orderId = intval($_GET['order_id']);
    $detailStmt = $conn->prepare("
        SELECT o.*, u.name AS customer_name, u.email AS customer_email
        FROM orders o
        LEFT JOIN users u ON o.user_id = u.id
        WHERE o.id = ?
    ");
    $detailStmt->bind_param('i', $orderId);
    $detailStmt->execute();
    $orderDetails = $detailStmt->get_result()->fetch_assoc();
    $detailStmt->close();

    if ($orderDetails) {
        $itemsStmt = $conn->prepare("
            SELECT * FROM order_items WHERE order_id = ?
        ");
        $itemsStmt->bind_param('i', $orderId);
        $itemsStmt->execute();
        $orderItems = $itemsStmt->get_result()->fetch_all(MYSQLI_ASSOC);
        $itemsStmt->close();
    }
}

require_once __DIR__ . '/../includes/header.php';
require_once __DIR__ . '/admin_nav.php';
?>

<div class="container mx-auto px-4 pb-12">
    <div class="grid lg:grid-cols-3 gap-8">
        <div class="lg:col-span-2 bg-white rounded-xl shadow-md border border-gray-200">
            <div class="flex items-center justify-between p-6 border-b border-gray-200">
                <div>
                    <h1 class="text-2xl font-display font-bold text-gray-900">Orders</h1>
                    <p class="text-sm text-gray-500">Manage customer orders and fulfillment</p>
                </div>
                <form method="GET" class="flex items-center gap-3">
                    <select name="status" class="input-field">
                        <option value="">All statuses</option>
                        <?php foreach ($allowedStatuses as $status): ?>
                            <option value="<?php echo $status; ?>" <?php echo $statusFilter === $status ? 'selected' : ''; ?>>
                                <?php echo ucfirst($status); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                    <button type="submit" class="btn-outline text-sm">Filter</button>
                </form>
            </div>

            <div class="overflow-x-auto">
                <table class="w-full text-sm">
                    <thead class="bg-gray-50">
                        <tr>
                            <th class="px-4 py-3 text-left font-semibold text-gray-600">Order</th>
                            <th class="px-4 py-3 text-left font-semibold text-gray-600">Customer</th>
                            <th class="px-4 py-3 text-left font-semibold text-gray-600">Total</th>
                            <th class="px-4 py-3 text-left font-semibold text-gray-600">Status</th>
                            <th class="px-4 py-3 text-right font-semibold text-gray-600">Actions</th>
                        </tr>
                    </thead>
                    <tbody class="divide-y divide-gray-100">
                        <?php while ($order = $orders->fetch_assoc()): ?>
                            <tr class="hover:bg-gray-50">
                                <td class="px-4 py-3">
                                    <div class="font-semibold text-gray-900">#<?php echo htmlspecialchars($order['order_number']); ?></div>
                                    <div class="text-xs text-gray-500">
                                        <?php echo date('M d, Y H:i', strtotime($order['created_at'])); ?>
                                    </div>
                                </td>
                                <td class="px-4 py-3 text-gray-600">
                                    <?php echo htmlspecialchars($order['customer_name'] ?? $order['shipping_name']); ?><br>
                                    <span class="text-xs text-gray-400"><?php echo htmlspecialchars($order['shipping_email']); ?></span>
                                </td>
                                <td class="px-4 py-3 font-semibold text-gray-900">
                                    <?php echo formatCurrency($order['total_amount']); ?>
                                </td>
                                <td class="px-4 py-3">
                                    <span class="badge <?php echo match($order['status']) {
                                        'delivered' => 'bg-green-100 text-green-800',
                                        'shipped' => 'bg-blue-100 text-blue-800',
                                        'processing' => 'bg-yellow-100 text-yellow-800',
                                        'cancelled' => 'bg-red-100 text-red-800',
                                        default => 'bg-gray-100 text-gray-800'
                                    }; ?>">
                                        <?php echo ucfirst($order['status']); ?>
                                    </span>
                                </td>
                                <td class="px-4 py-3 text-right">
                                    <a href="<?php echo SITE_URL; ?>/admin/orders.php?order_id=<?php echo $order['id']; ?>" class="btn-outline text-xs">
                                        <i class="fas fa-eye mr-1"></i>View
                                    </a>
                                </td>
                            </tr>
                        <?php endwhile; ?>
                    </tbody>
                </table>
            </div>
        </div>

        <div class="bg-white rounded-xl shadow-md border border-gray-200">
            <div class="p-6 border-b border-gray-200">
                <h2 class="text-xl font-display font-bold text-gray-900">Order Details</h2>
                <p class="text-sm text-gray-500">Select an order to view full details</p>
            </div>

            <div class="p-6">
                <?php if ($orderDetails): ?>
                    <div class="space-y-4">
                        <div>
                            <p class="text-sm text-gray-500">Order Number</p>
                            <p class="font-semibold text-gray-900">#<?php echo htmlspecialchars($orderDetails['order_number']); ?></p>
                        </div>
                        <div>
                            <p class="text-sm text-gray-500">Customer</p>
                            <p class="font-semibold text-gray-900"><?php echo htmlspecialchars($orderDetails['customer_name'] ?? $orderDetails['shipping_name']); ?></p>
                            <p class="text-sm text-gray-600"><?php echo htmlspecialchars($orderDetails['customer_email'] ?? $orderDetails['shipping_email']); ?></p>
                            <p class="text-sm text-gray-600"><?php echo htmlspecialchars($orderDetails['shipping_phone']); ?></p>
                        </div>
                        <div>
                            <p class="text-sm text-gray-500">Shipping Address</p>
                            <p class="text-sm text-gray-700 leading-relaxed">
                                <?php echo nl2br(htmlspecialchars($orderDetails['shipping_address'])); ?><br>
                                <?php echo htmlspecialchars($orderDetails['shipping_city']); ?>, <?php echo htmlspecialchars($orderDetails['shipping_postal_code']); ?>
                            </p>
                        </div>
                        <div>
                            <p class="text-sm text-gray-500">Status</p>
                            <span class="badge <?php echo match($orderDetails['status']) {
                                'delivered' => 'bg-green-100 text-green-800',
                                'shipped' => 'bg-blue-100 text-blue-800',
                                'processing' => 'bg-yellow-100 text-yellow-800',
                                'cancelled' => 'bg-red-100 text-red-800',
                                default => 'bg-gray-100 text-gray-800'
                            }; ?>">
                                <?php echo ucfirst($orderDetails['status']); ?>
                            </span>
                        </div>
                        <div>
                            <p class="text-sm text-gray-500">Total Amount</p>
                            <p class="text-lg font-semibold text-gray-900"><?php echo formatCurrency($orderDetails['total_amount']); ?></p>
                        </div>

                        <?php if (!empty($orderItems)): ?>
                            <div>
                                <p class="text-sm text-gray-500 mb-2">Items</p>
                                <ul class="space-y-2">
                                    <?php foreach ($orderItems as $item): ?>
                                        <li class="flex justify-between text-sm text-gray-700">
                                            <span><?php echo htmlspecialchars($item['product_name']); ?> × <?php echo $item['quantity']; ?></span>
                                            <span><?php echo formatCurrency($item['subtotal']); ?></span>
                                        </li>
                                    <?php endforeach; ?>
                                </ul>
                            </div>
                        <?php endif; ?>

                        <?php if ($orderDetails['notes']): ?>
                            <div>
                                <p class="text-sm text-gray-500 mb-1">Customer Notes</p>
                                <p class="text-sm text-gray-700 bg-gray-50 rounded-lg p-3">
                                    <?php echo nl2br(htmlspecialchars($orderDetails['notes'])); ?>
                                </p>
                            </div>
                        <?php endif; ?>
                    </div>
                <?php else: ?>
                    <div class="text-center text-gray-500 text-sm">
                        Select an order to view the full details.
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<?php
$orders->free();
closeDBConnection($conn);
require_once __DIR__ . '/../includes/footer.php';
?>
