<?php
require_once __DIR__ . '/../config/config.php';

if (!isLoggedIn()) {
    $_SESSION['admin_redirect'] = true;
    redirect(SITE_URL . '/login.php?redirect=admin');
} elseif (!isAdmin()) {
    redirect(SITE_URL . '/index.php');
}

$activePage = 'categories';
$pageTitle = 'Manage Categories | ' . SITE_NAME;

$conn = getDBConnection();
$errors = [];
$success = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    $categoryId = intval($_POST['category_id'] ?? 0);
    
    if (in_array($action, ['create', 'update'], true)) {
        $name = sanitize($_POST['name'] ?? '');
        $description = trim($_POST['description'] ?? '');
        $displayOrder = intval($_POST['display_order'] ?? 0);

        if (!$name) {
            $errors[] = 'Category name is required.';
        }

        $imageFileName = saveUploadedImage($_FILES['image'] ?? [], 'categories');

        if (!$errors) {
            $slug = slugify($name);

            if ($action === 'create') {
                $stmt = $conn->prepare("INSERT INTO categories (name, slug, description, image, display_order) VALUES (?, ?, ?, ?, ?)");
                $stmt->bind_param('ssssi', $name, $slug, $description, $imageFileName, $displayOrder);
                $stmt->execute();
                $stmt->close();
                $success = 'Category created successfully.';
            } elseif ($action === 'update' && $categoryId > 0) {
                if ($imageFileName) {
                    $stmt = $conn->prepare("UPDATE categories SET name = ?, slug = ?, description = ?, image = ?, display_order = ? WHERE id = ?");
                    $stmt->bind_param('ssssii', $name, $slug, $description, $imageFileName, $displayOrder, $categoryId);
                } else {
                    $stmt = $conn->prepare("UPDATE categories SET name = ?, slug = ?, description = ?, display_order = ? WHERE id = ?");
                    $stmt->bind_param('sssii', $name, $slug, $description, $displayOrder, $categoryId);
                }
                $stmt->execute();
                $stmt->close();
                $success = 'Category updated successfully.';
            }
        }
    } elseif ($action === 'delete' && $categoryId > 0) {
        $stmt = $conn->prepare("DELETE FROM categories WHERE id = ?");
        $stmt->bind_param('i', $categoryId);
        $stmt->execute();
        $stmt->close();
        $success = 'Category deleted successfully.';
    }
}

$categoriesResult = $conn->query("SELECT * FROM categories ORDER BY display_order ASC, name ASC");
$categories = $categoriesResult->fetch_all(MYSQLI_ASSOC);
$categoriesResult->close();

$editCategory = null;
if (isset($_GET['edit'])) {
    $editId = intval($_GET['edit']);
    foreach ($categories as $category) {
        if ($category['id'] == $editId) {
            $editCategory = $category;
            break;
        }
    }
}

require_once __DIR__ . '/../includes/header.php';
require_once __DIR__ . '/admin_nav.php';
?>

<div class="container mx-auto px-4 pb-12">
    <div class="max-w-3xl mx-auto">
        <div class="bg-white rounded-xl shadow-md border border-gray-200 p-6 mb-8">
            <div class="flex items-center justify-between mb-4">
                <h1 class="text-2xl font-display font-bold text-gray-900"><?php echo $editCategory ? 'Edit Category' : 'Create Category'; ?></h1>
                <?php if ($editCategory): ?>
                    <a href="<?php echo SITE_URL; ?>/admin/categories.php" class="text-sm text-primary-600 hover:text-primary-700">Cancel Edit</a>
                <?php endif; ?>
            </div>

            <?php if ($errors): ?>
                <div class="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-lg mb-4">
                    <ul class="list-disc list-inside text-sm space-y-1">
                        <?php foreach ($errors as $error): ?>
                            <li><?php echo $error; ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            <?php endif; ?>

            <?php if ($success): ?>
                <div class="bg-green-50 border border-green-200 text-green-700 px-4 py-3 rounded-lg mb-4">
                    <i class="fas fa-check-circle mr-2"></i><?php echo $success; ?>
                </div>
            <?php endif; ?>

            <form method="POST" enctype="multipart/form-data" class="space-y-4">
                <input type="hidden" name="action" value="<?php echo $editCategory ? 'update' : 'create'; ?>">
                <?php if ($editCategory): ?>
                    <input type="hidden" name="category_id" value="<?php echo $editCategory['id']; ?>">
                <?php endif; ?>

                <div>
                    <label class="block text-sm font-semibold text-gray-700 mb-2">Name</label>
                    <input type="text" name="name" class="input-field" value="<?php echo htmlspecialchars($editCategory['name'] ?? ''); ?>" required>
                </div>

                <div>
                    <label class="block text-sm font-semibold text-gray-700 mb-2">Description</label>
                    <textarea name="description" rows="3" class="input-field" placeholder="Optional">
                        <?php echo htmlspecialchars($editCategory['description'] ?? ''); ?>
                    </textarea>
                </div>

                <div class="grid md:grid-cols-2 gap-4">
                    <div>
                        <label class="block text-sm font-semibold text-gray-700 mb-2">Display Order</label>
                        <input type="number" name="display_order" class="input-field" value="<?php echo htmlspecialchars($editCategory['display_order'] ?? 0); ?>">
                    </div>
                    <div>
                        <label class="block text-sm font-semibold text-gray-700 mb-2">Image</label>
                        <input type="file" name="image" accept="image/jpeg,image/png,image/webp" class="block w-full text-sm text-gray-600">
                        <p class="text-xs text-gray-500 mt-1">Supported formats: JPG, PNG, WEBP. Max 5 MB.</p>
                    </div>
                </div>

                <div class="flex justify-end gap-3">
                    <button type="submit" class="btn-primary">
                        <i class="fas fa-save mr-2"></i>
                        <?php echo $editCategory ? 'Update Category' : 'Create Category'; ?>
                    </button>
                </div>
            </form>
        </div>

        <div class="bg-white rounded-xl shadow-md border border-gray-200 p-6">
            <div class="flex items-center justify-between mb-4">
                <h2 class="text-xl font-display font-bold text-gray-900">Categories</h2>
                <span class="text-sm text-gray-500"><?php echo count($categories); ?> total</span>
            </div>

            <div class="overflow-x-auto">
                <table class="w-full text-sm">
                    <thead class="bg-gray-50">
                        <tr>
                            <th class="px-4 py-3 text-left font-semibold text-gray-600">Category</th>
                            <th class="px-4 py-3 text-left font-semibold text-gray-600">Slug</th>
                            <th class="px-4 py-3 text-left font-semibold text-gray-600">Display Order</th>
                            <th class="px-4 py-3 text-right font-semibold text-gray-600">Actions</th>
                        </tr>
                    </thead>
                    <tbody class="divide-y divide-gray-100">
                        <?php foreach ($categories as $categoryRow): 
                            $categoryImage = getCategoryImage($categoryRow['image'] ?? null, $categoryRow['slug']);
                        ?>
                            <tr class="hover:bg-gray-50">
                                <td class="px-4 py-3">
                                    <div class="flex items-center gap-3">
                                        <div class="w-12 h-12 rounded-lg overflow-hidden bg-gray-100">
                                            <img src="<?php echo $categoryImage; ?>" alt="<?php echo htmlspecialchars($categoryRow['name']); ?>" class="w-full h-full object-cover">
                                        </div>
                                        <div>
                                            <p class="font-semibold text-gray-900"><?php echo htmlspecialchars($categoryRow['name']); ?></p>
                                            <?php if ($categoryRow['description']): ?>
                                                <p class="text-xs text-gray-500 line-clamp-2"><?php echo htmlspecialchars($categoryRow['description']); ?></p>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </td>
                                <td class="px-4 py-3 text-gray-600"><?php echo htmlspecialchars($categoryRow['slug']); ?></td>
                                <td class="px-4 py-3 text-gray-600"><?php echo intval($categoryRow['display_order']); ?></td>
                                <td class="px-4 py-3 text-right">
                                    <div class="flex items-center justify-end gap-2">
                                        <a href="<?php echo SITE_URL; ?>/admin/categories.php?edit=<?php echo $categoryRow['id']; ?>" class="btn-outline text-xs">
                                            <i class="fas fa-edit mr-1"></i>Edit
                                        </a>
                                        <form method="POST" onsubmit="return confirm('Delete this category?');">
                                            <input type="hidden" name="action" value="delete">
                                            <input type="hidden" name="category_id" value="<?php echo $categoryRow['id']; ?>">
                                            <button type="submit" class="btn-outline text-xs text-red-600 border-red-300 hover:bg-red-50">
                                                <i class="fas fa-trash mr-1"></i>Delete
                                            </button>
                                        </form>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>

<?php
closeDBConnection($conn);
require_once __DIR__ . '/../includes/footer.php';
?>
